<?php

// +-------------------------------------------------+
//  2002-2014 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: contribution_area_items.class.php,v 1.1.6.2 2025/08/08 13:03:28 qvarin Exp $

class contribution_area_items
{
    public const AVAILABLE_URI_TYPE = [
        contribution_area_item::RECORD_TYPE,
        contribution_area_item::SERIAL_TYPE,
        contribution_area_item::BULLETIN_TYPE,
        contribution_area_item::AUTHOR_TYPE,
        contribution_area_item::PUBLISHER_TYPE,
        contribution_area_item::WORK_TYPE,
        contribution_area_item::COLLECTION_TYPE,
        contribution_area_item::SUB_COLLECTION_TYPE,
        contribution_area_item::INDEXINT_TYPE,
        contribution_area_item::SERIE_TYPE,
        contribution_area_item::AUTHPERSO_TYPE,
        contribution_area_item::CONCEPT_TYPE,
        contribution_area_item::CATEGORY_TYPE
    ];

    /**
     *
     * @var contribution_area_store
     */
    private $store;

    /**
     *
     * @var string
     */
    private $uriType;

    /**
     *
     * @var int
     */
    private $nbEntities = 0;

    /**
     * Constructor
     *
     * @param string $uriType
     * @throws InvalidArgumentException
     */
    public function __construct(string $uriType)
    {
        if (!in_array($uriType, self::AVAILABLE_URI_TYPE)) {
            throw new InvalidArgumentException('Invalid uri type: ' . $uriType);
        }

        $this->uriType = $uriType;
        $this->store = new contribution_area_store();
    }

    /**
     * Fetch items from the store for the current type with identifier
     *
     * @return Generator|null
     */
    public function fetchItems(): ?Generator
    {
        $query = '
        SELECT * WHERE {
            ?uri rdf:type <' . $this->uriType . '> .
            ?uri pmb:identifier ?identifier .
        } ORDER BY ?uri
        ';

        $this->nbEntities = 0;
        $this->store->get_datastore()->query($query);
        if ($this->store->get_datastore()->num_rows()) {
            $this->nbEntities = $this->store->get_datastore()->num_rows();
            $results = $this->store->get_datastore()->get_result();
            foreach ($results as $result) {
                yield new contribution_area_item($result->uri);
            }
        }

        return null;
    }

    /**
     * Update display label items in the store for the current type
     *
     * @param bool $showProgression (optional)
     * @return bool
     */
    public function updateDisplayLabelItems(bool $showProgression = false)
    {
        $items = $this->fetchItems();
        if (null === $items) {
            if ($showProgression) {
                print ProgressBar::start(0, '[' . $this->uriType . '] pmb:displayLabel');
                print ProgressBar::finish();
            }
            return false;
        }

        $start = false;
        foreach ($items as $item) {
            if ($showProgression && !$start) {
                print ProgressBar::start($this->nbEntities, '[' . $this->uriType . '] pmb:displayLabel');
                $start = true;
            }

            $label = $item->computed_display_label($item->identifier);
            if (!empty($label)) {
                $item->update_display_label($label);
            }

            if ($showProgression && $start) {
                print ProgressBar::next();
            }
        }

        if ($showProgression && $start) {
            print ProgressBar::finish();
        }

        return true;
    }

    /**
     * Clean items in the store for the current type
     *
     * @param bool $showProgression (optional)
     * @return bool
     */
    public function cleanItems(bool $showProgression = false)
    {
        $items = $this->fetchItems();
        if (empty($items)) {
            if ($showProgression) {
                print ProgressBar::start(0, '[' . $this->uriType . '] clean');
                print ProgressBar::finish();
            }
            return false;
        }

        $start = false;
        foreach ($items as $item) {
            if ($showProgression && !$start) {
                print ProgressBar::start($this->nbEntities, '[' . $this->uriType . '] clean');
                $start = true;
            }

            // if ($item->entityExist($item->identifier)) {
                $query = 'DELETE {
                    <'. $item->uri .'> ?p ?o .
                } WHERE {
                    <'. $item->uri .'> ?p ?o .
                    <'. $item->uri .'> pmb:identifier ?identifier .
                    FILTER(?p != pmb:identifier && ?p != rdf:type && ?p != pmb:displayLabel)
                }';
            // } else {
            //     $query = 'DELETE {
            //         <'. $item->uri .'> ?p ?o .
            //     }';
            // }

            $this->store->get_datastore()->query($query);
            if ($showProgression && $start) {
                print ProgressBar::next();
            }
        }

        if ($showProgression && $start) {
            print ProgressBar::finish();
        }

        return true;
    }
}
