<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: SEOIndexationPagesController.php,v 1.4.2.2 2026/01/09 15:16:01 rtigero Exp $
namespace Pmb\SEO\Controller;

use Pmb\Common\Helper\Portal;
use Pmb\Common\Views\VueJsView;
use Pmb\SEO\Models\SEOPageType;
use Pmb\SEO\Models\SEOSubpageType;
use Pmb\SEO\Orm\SEOPageTypeOrm;
use Pmb\SEO\Orm\SEOSubpageTypeOrm;

class SEOIndexationPagesController extends SEOController
{
    /**
     * {@inheritDoc}
     * @see \Pmb\SEO\Controller\SEOController::defaultAction()
     */
    protected function defaultAction()
    {
        $viewData = array_merge($this->getViewBaseData(), SEOPageType::getData(), SEOSubpageType::getData());
        $view = new VueJsView("seo/indexationPages", $viewData);
        print $view->render();
    }

    /**
     * Sauvegarde les donnees
     * @return void
     */
    public function save()
    {
        global $msg;

        if (
            empty($this->data) ||
            !isset($this->data->pageTypes) ||
            !isset($this->data->pageSubTypes)
        ) {
            $this->ajaxError($msg['common_failed_save']);
            return;
        }

        // Supprime les donnees existantes
        SEOPageType::deleteAll();
        SEOSubpageType::deleteAll();

        // Ajoute les nouvelles donnees
        foreach ($this->data->pageTypes as $pageType) {
            $SEOPageType = new SEOPageType();
            $SEOPageType->setFromForm($pageType);
            $SEOPageType->create();

            foreach ($this->data->pageSubTypes as $pageSubType) {
                // Si le type de page n'est pas personnalise, on continue
                if (!isset($pageSubType->customized) || $pageSubType->customized == false) {
                    continue;
                }

                $subTypeParent = Portal::getTypeFromSubType($pageSubType->type);

                // Si le type de page est le meme que le type de page parent, on continue
                if ($subTypeParent === intval($SEOPageType->type)) {
                    $SEOSubpageType = new SEOSubpageType();
                    $SEOSubpageType->numPageType = $SEOPageType->id;
                    $SEOSubpageType->setFromForm($pageSubType);
                    $SEOSubpageType->create();
                }
            }
        }

        $this->ajaxJsonResponse(['success' => true]);
    }

    /**
     * Recupere les valeurs indexable et followable pour un type/sous-type de page
     * @param int $type Type de page
     * @param int|null $subtype Sous-type de page (optionnel)
     * @return array
     */
    public static function getIndexationPage($type, $subtype = null)
    {
        // Verifier si le sous-type existe en BDD (personnalise)
        if (!empty($subtype)) {
            $subpageOrms = SEOSubpageTypeOrm::finds(['type' => $subtype]);
            if (!empty($subpageOrms)) {
                $subpageOrm = reset($subpageOrms);
                return [
                    'indexable' => $subpageOrm->indexable,
                    'followable' => $subpageOrm->followable
                ];
            }
        }

        // Verifier si le type de page existe en BDD (personnalise)
        $pageOrms = SEOPageTypeOrm::finds(['type' => $type]);
        if (!empty($pageOrms)) {
            $pageOrm = reset($pageOrms);
            return [
                'indexable' => $pageOrm->indexable,
                'followable' => $pageOrm->followable
            ];
        }

        // Utiliser les valeurs par defaut depuis Portal
        $defaultRules = Portal::DEFAULT_PAGE_INDEXING_RULES;
        $pages = Portal::PAGES;

        // Trouver le label correspondant au type
        $typeLabel = array_search($type, $pages);
        if ($typeLabel !== false && isset($defaultRules[$typeLabel])) {
            return [
                'indexable' => $defaultRules[$typeLabel]['indexable'],
                'followable' => $defaultRules[$typeLabel]['followable']
            ];
        }

        // Valeurs par defaut si rien n'est trouve
        return [
            'indexable' => false,
            'followable' => false
        ];
    }
}
