<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: Scenario.php,v 1.24.2.2 2025/06/27 14:20:55 rtigero Exp $

namespace Pmb\ImportExport\Models;

use encoding_normalize;
use Pmb\Common\Helper\Helper;
use Pmb\Common\Models\Model;
use Pmb\ImportExport\Models\Sources\Source;
use Pmb\ImportExport\Models\Steps\Step;
use Pmb\ImportExport\Orm\SourceOrm;
use Pmb\ImportExport\Orm\StepOrm;
use stdClass;

class Scenario extends Model
{
    public $id = 0;
    public $idScenario = 0;
    public $scenarioName = "";
    public $scenarioComment = "";
    public $scenarioSettings = null;

    public $sources = array();
    public $steps = array();

    protected $runMode = false;

    protected $ormName = "Pmb\ImportExport\Orm\ScenarioOrm";

    public function __construct(int $id = 0, bool $runMode = false)
    {
        $this->runMode = $runMode;
        parent::__construct($id);
    }

    protected function fetchData()
    {
        parent::fetchData();
        $this->fetchSources();
        $this->fetchSteps();
    }

    protected function fetchSources()
    {
        $sourcesOrm = SourceOrm::finds(["num_scenario" => $this->id]);
        $this->sources = array();
        foreach ($sourcesOrm as $sourceOrm) {
            if (class_exists($sourceOrm->source_type)) {
                $source = new $sourceOrm->source_type($sourceOrm->id_source, $this, $this->runMode);
            } else {
                $source = new Source($sourceOrm->id_source, $this, $this->runMode);
            }
            $settings = Helper::toArray(encoding_normalize::json_decode($sourceOrm->source_settings));
            if (is_array($settings)) {
                $source->setBaseParameters($settings);
            }
            $this->sources[] = $source;
        }
    }

    /**
     * Recupere la source d'un scenario par son id
     *
     * @param integer $sourceId
     * @return void
     */
    public function getSourceById(int $sourceId = 0)
    {
        if (!$sourceId || empty($this->sources)) {
            return null;
        }

        $foundSource = null;
        foreach ($this->sources as $k => $source) {
            if ($source->id == $sourceId) {
                $foundSource = $this->sources[$k];
                break;
            }
        }
        return $foundSource;
    }

    protected function fetchSteps()
    {
        $stepsOrm = StepOrm::finds(["num_scenario" => $this->id]);
        $this->steps = array();
        foreach ($stepsOrm as $stepOrm) {
            if (class_exists($stepOrm->step_type)) {
                $step = new $stepOrm->step_type($stepOrm->id_step, $this, $this->runMode);
            } else {
                $step = new Step($stepOrm->id_step, $this, $this->runMode);
            }
            $this->steps[] = $step;
        }
    }

    /**
     * Recupere l'etape d'un scenario par son id
     *
     * @param integer $stepId
     * @return void
     */
    public function getStepById(int $stepId = 0)
    {
        if (!$stepId || empty($this->steps)) {
            return null;
        }

        $foundStep = null;
        foreach ($this->steps as $k => $step) {
            if ($step->id == $stepId) {
                $foundStep = $this->steps[$k];
                break;
            }
        }
        return $foundStep;
    }

    public function setFromForm(object $data)
    {
        $this->scenarioName = $data->scenarioName ?? "";
        $this->scenarioComment = $data->scenarioComment ?? "";
    }

    public function save()
    {
        $orm = new $this->ormName($this->id);

        $orm->scenario_name = $this->scenarioName;
        $orm->scenario_comment = $this->scenarioComment;

        $orm->save();
        if (!$this->id) {
            $this->id = $orm->id_scenario;
        }
        return $orm;
    }

    public function remove()
    {
        $orm = new $this->ormName($this->id);

        foreach ($this->steps as $step) {
            $step->remove();
        }
        foreach ($this->sources as $source) {
            $source->remove();
        }
        $orm->delete();
    }

    public function addSource($source)
    {
        $this->sources[] = $source;
    }

    public function addStep($step)
    {
        $this->steps[] = $step;
    }

    public function execute($scenarioData = null)
    {
        // Pour executer les steps dans l'ordre !!
        usort($this->steps, function ($a, $b) {
            return $a->stepOrder > $b->stepOrder;
        });
        foreach ($this->steps as $step) {
            $step->execute($scenarioData->sources);
        }
    }

    public function duplicate()
    {
        $newScenario = clone $this;

        $newScenario->id = 0;
        $newScenario->scenarioName .= " - copy";
        $newScenario->save();
        $newSources = array();
        $newSteps = array();

        foreach ($this->sources as $source) {
            $source->numScenario = $newScenario->id;
            $newSources[] = $source->duplicate();
        }

        foreach ($this->steps as $step) {
            //Gestion des sources associees aux synchros
            if (!empty($step->stepSettings) && !empty($step->stepSettings->source)) {
                $sourceIndex = array_search($step->stepSettings->source, array_column($this->sources, 'id'));
                if ($sourceIndex !== false) {
                    $step->stepSettings->source = $newSources[$sourceIndex]->id;
                }
            }
            $step->numScenario = $newScenario->id;
            $newSteps[] = $step->duplicate();
        }
        $newScenario->sources = $newSources;
        $newScenario->steps = $newSteps;

        return $newScenario;
    }

    public function import(stdClass $data)
    {
        $this->scenarioName = $data->scenarioName ?? "";
        $this->scenarioComment = $data->scenarioComment ?? "";
        $this->scenarioSettings = $data->scenarioSettings ?? "";
        $this->save();

        if (!$this->id) {
            return;
        }
        //Tableau de type ancien id => nouveau
        $sourceIds = array();

        //Import des sources
        foreach ($data->sources as $source) {
            $newSource = new Source();
            $newSource->import($source, $this->id);
            $sourceIds[$source->id] = $newSource->id;
            $this->addSource($newSource);
        }

        //Import des steps
        foreach ($data->steps as $step) {
            $newStep = new Step();
            $newStep->import($step, $this->id, $sourceIds);
            $this->addStep($newStep);
        }
    }
}
