<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: OntologyPMB.php,v 1.12.2.3 2025/07/25 13:48:50 rtigero Exp $

namespace Pmb\ImportExport\Models\Ontology;

use encoding_normalize;
use marc_list_collection;
use onto_parametres_perso;
use Pmb\Common\Helper\GlobalContext;
use Pmb\ImportExport\Models\Sources\SourceFile\FileSystem\FileSystemLocal\FileSystemUploaded\FileSystemUploaded;

class OntologyPMB extends ImportExportOntology
{
    private static $instance = null;

    private const STORE_NAME = "ontology_pmb_entities";

    public static function getInstance()
    {
        if (is_null(static::$instance)) {
            static::$instance = new self();
        }
        return static::$instance;
    }

    private function __clone() {}

    private function __wakeup() {}

    private function __construct()
    {
        $this->store = new Store(self::STORE_NAME);
        $this->loadParametresPerso();
        $this->fetchData();
    }

    protected function fetchData()
    {
        //On rcupre toutes les entits PMB
        $query = "SELECT ?entity ?label ?name ?comment WHERE {
            ?entity pmb:flag 'pmb_entity' ;
                rdfs:label ?label ;
                rdfs:comment ?comment ;
                pmb:name ?name .
        } ORDER BY ASC(?label)";
        $result = $this->store->query($query);
        if (!is_countable($result)) {
            throw new \Exception("Unable to load PMB ontology");
        }
        foreach ($result as $row) {
            $entity = $this->getEntityByURI($row["entity"]);

            $entity->name = $row["name"];
            $entity->displayLabel = $row["label"];
            $entity->comment = $row["comment"];

            //On rcupre toutes les proprits associes  l'entit
            $queryProperties = "SELECT * WHERE {
                ?propURI rdfs:domain <" . $row["entity"] . "> ;
                    rdf:type rdf:Property ;
                    pmb:name ?name ;
                    pmb:datatype ?datatype ;
                    rdfs:label ?label .
                    OPTIONAL {
                       ?propURI rdfs:range ?range .
                    }
                    OPTIONAL {
                       ?propURI pmb:flag ?flag .
                    }
                    OPTIONAL {
                       ?propURI rdfs:comment ?comment .
                    }
                    OPTIONAL {
                        ?propNode owl:onProperty ?propURI ;
                            owl:maxCardinality ?maxCardinality .
                        OPTIONAL {
                            ?propNode owl:minCardinality ?minCardinality .
                        }
                    }
            } ORDER BY ASC(?label)";
            $resultProp = $this->store->query($queryProperties);
            if (is_countable($resultProp)) {
                foreach ($resultProp as $rowProp) {
                    $property = $this->getPropertyByURI($rowProp["propURI"]);
                    $property->name = $rowProp["name"];
                    $property->displayLabel = $rowProp["label"];
                    if (isset($rowProp["comment"])) {
                        $property->comment = $rowProp["comment"];
                    }
                    if (isset($rowProp["range"])) {
                        $range = $this->getPropertyByURI($rowProp["range"]);
                        $property->addRange($range);
                    }
                    if (isset($rowProp["flag"])) {
                        $property->flag = $rowProp["flag"];
                    }
                    //On postule que la proprit est multiple par dfaut
                    $property->maxCardinality = "n";
                    if (isset($rowProp["maxCardinality"])) {
                        if ($rowProp["maxCardinality"] != "n") {
                            $property->maxCardinality = intval($rowProp["maxCardinality"]);
                        }
                    }
                    //On postule que la proprit est optionnelle par dfaut
                    $property->minCardinality = 0;
                    if (isset($rowProp["minCardinality"])) {
                        $property->minCardinality = intval($rowProp["minCardinality"]);
                    }
                    $this->getInputDataFromDatatype($property, $rowProp["datatype"]);
                    $this->addProperty($property);
                    $entity->addProperty($property);
                }
                $this->addEntity($entity);
            }
        }
    }
    /**
     * Ajoute  la proprit data de $property un tableau reprsentant les informations sur le type d'input
     * et les ventuelles donnes associes
     *
     * @param Property $property
     * @param string $datatype
     *
     * @return void
     */

    protected function getInputDataFromDatatype(&$property, $datatype)
    {
        switch ($datatype) {
            case "http://www.pmbservices.fr/ontology#integer":
                $property->data = [
                    "inputType" => "input",
                    "type" => "number"
                ];
                break;
            case "http://www.pmbservices.fr/ontology#date":
                $property->data = [
                    "inputType" => "input",
                    "type" => "date"
                ];
                break;
            case "http://www.pmbservices.fr/ontology#small_text":
                $property->data = [
                    "inputType" => "input",
                    "type" => "text"
                ];
                break;
            case "http://www.pmbservices.fr/ontology#small_text_link":
                $property->data = [
                    "inputType" => "input",
                    "type" => "url"
                ];
                break;
            case "http://www.pmbservices.fr/ontology#text":
                $property->data = [
                    "inputType" => "textarea"
                ];
                break;
            case "http://www.pmbservices.fr/ontology#boolean":
                $property->data = [
                    "inputType" => "input",
                    "type" => "checkbox"
                ];
                break;
            case "http://www.pmbservices.fr/ontology#file":
                //TODO gestion des fichiers
                // $property->data = [
                //     "inputType" => "input",
                //     "type" => "file"
                // ];
                break;
            case "http://www.pmbservices.fr/ontology#marclist":
                //On rcupre la marclist ici
                $values = array();
                $query = "SELECT ?marclistType WHERE {
                    <" . $property->uri . "> pmb:marclist_type ?marclistType .
                }";
                $result = $this->store->query($query);
                if (count($result)) {
                    $type = $result[0]["marclistType"];
                    $marclist = marc_list_collection::get_instance($type);
                    if ($marclist->table) {
                        foreach ($marclist->table as $key => $value) {
                            $values[] = [
                                "value" => $key,
                                "label" => $value
                            ];
                        }
                    }
                }
                $property->data = [
                    "inputType" => "select",
                    "options" => $values
                ];
                break;
            case "http://www.pmbservices.fr/ontology#list":
                //On rcupre la requte pour rcuperer la liste
                $values = array();
                $query = "SELECT * WHERE {
                    <" . $property->uri . "> rdf:type ?type .
                    OPTIONAL {
                        <" . $property->uri . "> pmb:list_query ?listQuery .
                    }
                    OPTIONAL {
                        <" . $property->uri . "> pmb:list_item ?listItem .
                        ?listItem rdfs:label ?label ;
                            pmb:identifier ?value .
                    }
                }";
                $result = $this->store->query($query);
                if (! count($result)) {
                    break;
                }
                //Cas ou on a une requete  jouer
                if (count($result) == 1 && array_key_exists("listQuery", $result[0])) {
                    $query = $result[0]["listQuery"];
                    $result = pmb_mysql_query($query);
                    if (pmb_mysql_num_rows($result)) {
                        while ($row = pmb_mysql_fetch_array($result)) {
                            //Normalement le premier lment de la requte est l'id et le second est le label
                            $values[] = [
                                "value" => $row[0],
                                "label" => $row[1]
                            ];
                        }
                    }
                } else {
                    //Cas ou on a des items
                    foreach ($result as $row) {
                        $values[] = [
                            "value" => $row["value"],
                            "label" => $row["label"]
                        ];
                    }
                }
                $property->data = [
                    "inputType" => "select",
                    "options" => $values
                ];
                break;
            case "http://www.pmbservices.fr/ontology#upload_directories":
                $property->data = [
                    "inputType" => "select",
                    "options" => FileSystemUploaded::getPMBUploadFolders()
                ];
                break;
            case "http://www.pmbservices.fr/ontology#licence":
            case "http://www.pmbservices.fr/ontology#resolve":
            case "http://www.pmbservices.fr/ontology#floating_date":
            case "http://www.pmbservices.fr/ontology#multilingual_qualified":
                //TODO
                break;
            case "http://www.pmbservices.fr/ontology#resource":
            case "http://www.pmbservices.fr/ontology#resource_selector":
            case "http://www.pmbservices.fr/ontology#responsability_selector":
                //On doit rcuprer le type de ressource concerne
                $query = "SELECT * WHERE {
                    <" . $property->uri . "> rdfs:range ?range .
                    OPTIONAL {
                        <" . $property->uri . "> pmb:cp_options ?cpOptions .
                    }
                }";
                $result = $this->store->query($query);
                $type = explode("#", $result[0]["range"]);
                $type = $type[1];
                $attIdFilter = "";
                $cpOptions = [];
                if(isset($result[0]["cpOptions"])) {
                    $cpOptions = encoding_normalize::json_decode($result[0]["cpOptions"], true);
                }
                switch ($type) {
                    case "linked_record":
                    case "record":
                        $type = "notices";
                        break;
                    case "bulletin":
                        $type = "bull";
                        break;
                    case "responsability":
                        $type = "authors";
                        break;
                    case "category":
                        $type = "categories";
                        break;
                    case "indexint":
                        $type = "indexint_mul";
                        break;
                    case "ConceptScheme":
                    case "Concept":
                        $type = "onto";
                        if(is_countable($property->ranges) && count($property->ranges)) {
                            $attIdFilter = $property->ranges[array_keys($property->ranges)[0]]->uri;
                        }
                        break;
                    case "work":
                        $type = "titre_uniforme";
                        break;
                    default:
                        if (stripos($type, "authperso") === false) {
                            $type .= "s";
                        }
                        break;
                }
                $property->data = [
                    "inputType" => "resource",
                    'ajax' => $type,
                    'selector' => $type,
                    'p1' => "",
                    'p2' => 1,
                    'att_id_filter' => $attIdFilter,
                    'cp_options' => $cpOptions
                ];
                break;
            case "http://www.pmbservices.fr/ontology#marclist_selector": //Pas vu d'utilisation de ce datatype donc je l'ai mis ici: a surveiller !
            case "http://www.pmbservices.fr/ontology#marclist_type": //IDEM
            case "http://www.pmbservices.fr/ontology#list_item": //IDEM
            case "http://www.pmbservices.fr/ontology#list_query": //IDEM
            case "http://www.pmbservices.fr/ontology#merge_selector": //IDEM
            default:
                break;
        }
    }

    /**
     * Charge les paramtres perso dans l'ontologie
     *
     * @return void
     */
    protected function loadParametresPerso()
    {
        $reset = $this->store->load(GlobalContext::get("class_path") . "/rdf/ontologies_pmb_entities.rdf", onto_parametres_perso::is_modified());
        onto_parametres_perso::load_in_store($this->store, $reset);
    }
}
