<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: ChunkSpreadsheet.php,v 1.2.2.2 2025/06/27 14:21:03 rtigero Exp $

namespace Pmb\ImportExport\Models\Chunks\ChunkSpreadsheet;

use encoding_normalize;
use Pmb\ImportExport\Models\Chunks\Chunk;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Reader\IReader as IREADER;

class ChunkSpreadsheet extends Chunk
{

    protected $sheet = '';
    protected $headerLine = 0;
    protected $startLine = 0;
    protected $headers = [];

    public const ENCLOSURE = '||';
    public const SEPARATOR = ';';

    public function __construct($resource, $parameters)
    {
        parent::__construct($resource, $parameters);
        $this->parameters['enclosure'] = static::ENCLOSURE;
        $this->parameters['separator'] = static::SEPARATOR;
    }

    protected function initialize()
    {
        if (!$this->isInitialized) {
            $this->isInitialized = true;

            $this->initializeStream();
            $sheet = $this->parameters['sheet'] ?? 0;
            $sheet = trim($sheet);
            $sheet = ('' === $sheet) ? 0 : $sheet;
            $this->sheet = $sheet;
            $this->headerLine = intval($this->parameters["headerline"] ?? 0);
            $this->startLine = intval($this->parameters["startline"] ?? 0);
        }
    }

    public function next()
    {
        $this->initialize();

        $type = null;
        $spreadsheet = null;
        $worksheet = null;
        try {

            $type = IOFactory::identify($this->uri);
            $reader = IOFactory::createReader($type);
            $spreadsheet = $reader->load($this->uri, IReader::READ_DATA_ONLY);
            switch (true) {

                case (is_numeric($this->sheet)):
                    $worksheet = $spreadsheet->setActiveSheetIndex($this->sheet);
                    break;

                case (is_string($this->sheet)):
                    $worksheet = $spreadsheet->setActiveSheetIndexByName($this->sheet);
                    break;

                default:
                    $worksheet = $spreadsheet->setActiveSheetIndex(0);
                    break;
            }

            $endRow = $worksheet->getHighestDataRow();
            $endColumn = $worksheet->getHighestDataColumn();

            $rowIterator = $spreadsheet->getActiveSheet()->getRowIterator(1, $endRow);

            // Construction de l'entete si on a un numero de ligne pour le header
            $header = [];
            if ($this->headerLine) {

                $row = $rowIterator->seek($this->headerLine)->current();
                foreach ($row->getCellIterator('A', $endColumn) as $cell) {
                    $header[] = $cell->getValue();
                }

                if (is_array($header)) {
                    array_walk($header, function (&$value, $key) {
                        $value = encoding_normalize::charset_normalize($value, encoding_normalize::detect_encoding($value, ["UTF-8", "ISO-8859-1", "ISO-8859-15", "cp1252"]));
                        $value = strip_empty_chars($value);
                        $value = preg_replace("/\s/", "_", $value);
                        if ($value == "") {
                            $value = "col_" . (intval($key) + 1);
                        }
                    });
                    $this->headers = $header;
                }
            }

            // Placement sur la ligne de debut ou sur la ligne apres la ligne d'entete
            if ($this->startLine) {
                $rowIterator->seek($this->startLine);
            } elseif ($this->headerLine) {
                $rowIterator->next();
            }

            $currentContent = '';
            $enclosure = $this->parameters['enclosure'] ?? static::ENCLOSURE;
            $separator = $this->parameters['separator'] ?? static::SEPARATOR;
            $glue = $enclosure . $separator . $enclosure;

            while ($rowIterator->valid()) {

                $row = $rowIterator->current();

                if (!$row->isEmpty()) {

                    $rowContent = [];
                    $trimmedRowContent = '';

                    foreach ($row->getCellIterator('A', $endColumn) as $cell) {
                        $rowContent[] = $cell->getValue() ?? '';
                        $trimmedRowContent .= trim($cell->getValue() ?? '');
                    }

                    if(!empty($trimmedRowContent)) {
                        if (count($this->headers)) {
                            $currentContent = $enclosure . implode($glue, $this->headers) . $enclosure . "\r\n";
                        }
                        $currentContent .= $enclosure . implode($glue, $rowContent) . $enclosure . "\r\n";
                        yield $currentContent;
                        $currentContent = '';
                    }

                }
                $rowIterator->next();
            }

        } catch (\Exception $e) {
            yield null;
        }
    }

}
