<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: Image.php,v 1.9.2.2.2.1.2.1.2.4 2025/12/02 15:52:53 jparis Exp $

namespace Pmb\Common\Library\Image;

use Pmb\Common\Helper\GlobalContext;
use Pmb\Common\Helper\HTML;

class Image
{
    public const MIMETYPE = [
        "image/jpeg",
        "image/jpg",
        "image/x-png",
        "image/png",
        "image/gif",
        "image/webp",
    ];

    public static function isValid(string $image)
    {
        $finfo = new \finfo();
        $mimeType = $finfo->buffer($image, FILEINFO_MIME_TYPE);

        $img = @imagecreatefromstring($image);
        return in_array($mimeType, static::MIMETYPE, true) && !empty($img);
    }

    public static function resize(string $image, ?int $maxSize = null)
    {
        if (empty($image) || !static::isValid($image)) {
            return null;
        }

        if (!$maxSize) {
            $maxSize = GlobalContext::get("notice_img_pics_max_size") ?? (GlobalContext::get("pmb_notice_img_pics_max_size") ?? 150);
        }

        $img = imagecreatefromstring($image);
        if (empty($img)) {
            return null;
        }

        $redim = false;
        if (imagesx($img) >= imagesy($img)) {
            if (imagesx($img) <= $maxSize) {
                $largeur = imagesx($img);
                $hauteur = imagesy($img);
            } else {
                $redim = true;
                $largeur = $maxSize;
                $hauteur = ($largeur * imagesy($img)) / imagesx($img);
            }
        } else {
            if (imagesy($img) <= $maxSize) {
                $hauteur = imagesy($img);
                $largeur = imagesx($img);
            } else {
                $redim = true;
                $hauteur = $maxSize;
                $largeur = ($hauteur * imagesx($img)) / imagesy($img);
            }
        }

        // Convertion float en int
        $largeur = intval($largeur);
        $hauteur = intval($hauteur);

        // Verification de la taille minimum requis
        $largeur = ($largeur < 1) ? 1 : $largeur;
        $hauteur = ($hauteur < 1) ? 1 : $hauteur;

        // Creation de l'image
        $imgResized = imagecreatetruecolor($largeur, $hauteur);

        // On active la transparence
        imageSaveAlpha($imgResized, true);
        imageAlphaBlending($imgResized, false);
        $transparent = imagecolorallocatealpha($imgResized, 255, 255, 255, 127);
        imagefill($imgResized, 0, 0, $transparent);

        if ($redim) {
            imagecopyresampled($imgResized, $img, 0, 0, 0, 0, $largeur, $hauteur, imagesx($img), imagesy($img));
        } else {
            imagecopyresampled($imgResized, $img, 0, 0, 0, 0, $largeur, $hauteur, $largeur, $hauteur);
        }
        imagedestroy($img);
        return $imgResized;
    }

    public static function format(string $image, ?int $maxSize = null, string $watermark = "")
    {
        $image = Image::resize($image, $maxSize);
        if (empty($image)) {
            return null;
        }

        // Copyright
        if (!empty($watermark)) {
            $white = imagecolorallocate($image, 255, 255, 255);
            imagestring($image, 1, intval((imagesx($image) / 3)), intval((imagesy($image) / 1.1)), $watermark, $white);
        }

        return $image;
    }

    public static function print($image)
    {
        switch (GlobalContext::get("img_cache_type")) {
            case "png":
                return static::printPNG($image);

            case "webp":
            default:
                return static::printWebP($image);
        }
    }

    public static function printPNG($image)
    {
        if (empty($image)) {
            return null;
        }

        header('Content-Type: image/png');
        imagepng($image, null, 9, defined('PNG_ALL_FILTERS') ? PNG_ALL_FILTERS : null);
        imagedestroy($image);
        return true;
    }

    public static function printSVG($image)
    {
        if (empty($image)) {
            return null;
        }

        header('Content-Type: image/svg+xml');
        print $image;
        return true;
    }

    public static function printWebP($image)
    {
        if (empty($image)) {
            return null;
        }

        header('Content-Type: image/webp');
        imagewebp($image);
        imagedestroy($image);
        return true;
    }

    /**
     * Recupere les imges pour y ajouter l'attribut loading="lazy"
     *
     * @param string $html
     * @return string
     */
    public static function transformHTML(string $html): string
    {
        $html = HTML::cleanHTML($html);
        if('' == $html) {
            return '';
        }
        $dom = new \DOMDocument();
        if (@$dom->loadHTML($html)) {

            $imgList = $dom->getElementsByTagName("img");
            for ($i = 0; $i < $imgList->length; $i++) {

            	/**
                 * @var \DOMElement $img
                 */
                $img = $imgList->item($i);
                if (!$img->hasAttribute("loading")) {
                    $img->setAttribute("loading", "lazy");
                }
            }

            return $dom->saveHTML();
        }
        return $html;
    }

    /**
     * Convertion de l'image webp en png
     *
     * @param string $binary Contenu de l'image
     * @return array<int, string>|false Tableau contenant le binary et le mime type
     */
    public static function convertBinaryWebpToPng(string $binary)
    {
        $mime_content_type = finfo_buffer(finfo_open(), $binary, FILEINFO_MIME_TYPE);
        if (!in_array($mime_content_type, static::MIMETYPE, true)) {
            return false;
        }

        if ($mime_content_type === 'image/webp') {
            global $base_path;

            // Convertion de l'image webp en png
            $temp_img = $base_path . '/temp/tmp_img_'. md5(time()) .'.png';
            imagepng(imagecreatefromstring($binary), $temp_img);

            // Recuperation de l'image
            $binary = file_get_contents($temp_img);
            $mime_content_type = mime_content_type($temp_img);

            // Suppression de l'image temp
            unlink($temp_img);

            return [$binary, $mime_content_type];
        }

        // L'image n'est pas un webp on la retourne
        return [$binary, $mime_content_type];
    }

    /**
     * Compression d'une image
     *
     * @param string $from Chemin de l'image source
     * @param string $to Chemin de l'image de destination
     * @param integer $quality Qualit de compression (1-100) (default: 50)
     * @return void
     */
    public static function setCompressionQuality(string $from, string $to, int $quality = 50): void
    {
        if (false === is_file($from) || !self::isValid(file_get_contents($from))) {
            throw new \Exception('Image file not found');
        }

        if (false === extension_loaded('imagick')) {
            trigger_error('Extension imagick not loaded', E_USER_NOTICE);
            file_put_contents($to, file_get_contents($from));
            return;
        }

        if ($quality <= 0 || 100 < $quality) {
            throw new \Exception("Quality must be between 1 and 100");
        }

        $backgroundImagick = new \Imagick($from);
        $imagick = new \Imagick();
        $imagick->setCompressionQuality($quality);
        $imagick->newPseudoImage(
            $backgroundImagick->getImageWidth(),
            $backgroundImagick->getImageHeight(),
            'canvas:white'
        );
        $imagick->compositeImage($backgroundImagick, \Imagick::COMPOSITE_ATOP, 0, 0);

        $ext = pathinfo($from, PATHINFO_EXTENSION);
        if (empty($ext)) {
            $finfo = new \finfo();
            $ext = $finfo->buffer(file_get_contents($from), FILEINFO_EXTENSION);
        }

        switch ($ext) {
            case 'png':
            case 'webp':
                $imagick->setFormat("png");
                break;

            case 'jpg':
            case 'jpeg':
                $imagick->setFormat("jpg");
                break;

            default:
                throw new \Exception("Unsupported image format");
        }

        if (is_file($to)) {
            unlink($to);
        }
        file_put_contents($to, $imagick->getImageBlob());
    }
    
    /**
     * Convertit une image base64 vers un autre format en base64
     *
     * @param string $base64Image Image en base64 (avec ou sans data URI)
     * @param string $outputFormat Format de sortie : 'png' ou 'jpeg'
     * @param int $quality Qualite JPEG (1-100), ignore pour PNG
     * @return string Image convertie en base64 avec data URI
     */
    public static function convertBase64Image($base64Image, $outputFormat = 'jpeg', $quality = 90)
    {
        // Valider le format de sortie
        $outputFormat = strtolower($outputFormat);
        if (!in_array($outputFormat, ['png', 'jpeg', 'jpg'])) {
            throw new \Exception("Invalid output format");
        }
        
        // Normaliser jpeg/jpg
        if ($outputFormat === 'jpg') {
            $outputFormat = 'jpeg';
        }
        
        // Nettoyer la chaine base64 (enlever le data URI si present)
        $base64Clean = self::cleanBase64String($base64Image);
        
        // Decoder le base64
        $imageData = base64_decode($base64Clean);
        if ($imageData === false) {
            throw new \Exception("Unable to decode base64");
        }
        
        // Creer une ressource image depuis la chaine
        $sourceImage = imagecreatefromstring($imageData);
        if ($sourceImage === false) {
            throw new \Exception("Unsupported image format");
        }
        
        // Convertir selon le format de sortie
        ob_start();
        if ($outputFormat === 'png') {
            // Preserver la transparence pour PNG
            imagealphablending($sourceImage, false);
            imagesavealpha($sourceImage, true);
            imagepng($sourceImage, null, 9);
            $mimeType = 'image/png';
        } else {
            imagejpeg($sourceImage, null, $quality);
            $mimeType = 'image/jpeg';
        }
        $imageOutput = ob_get_clean();
        
        // Liberer la memoire
        imagedestroy($sourceImage);
        
        // Encoder en base64 avec data URI
        $base64Output = base64_encode($imageOutput);
        return "data:$mimeType;base64,$base64Output";
    }
    
    /**
     * Nettoie une chaine base64 en enlevant le data URI
     *
     * @param string $base64String
     * @return string
     */
    public static function cleanBase64String($base64String)
    {
        // Enlever les espaces et sauts de ligne
        $base64String = preg_replace('/\s+/', '', $base64String);
        
        // Si c'est un data URI, extraire juste la partie base64
        if (preg_match('/^data:image\/[a-z]+;base64,(.+)$/i', $base64String, $matches)) {
            return $matches[1];
        }
        
        return $base64String;
    }
    
    /**
     * Convertit toutes les images d'un HTML en base64 avec conversion de format optionnelle
     *
     * @param string $html
     * @param string|null $outputFormat Format de sortie : 'jpeg', 'png' ou null pour conserver le format original
     * @param int $quality Qualite de compression JPEG (1-100, ignore pour PNG), defaut: 90
     * @return string
     */
    public static function convertHTMLImagesToBase64(string $html, ?string $outputFormat = null, int $quality = 90): string
    {
        // Valider le format de sortie
        $outputFormat = strtolower($outputFormat);
        if (!in_array($outputFormat, ['png', 'jpeg', 'jpg']) && $outputFormat !== null) {
            throw new \Exception("Invalid output format");
        }
        
        // Trouver toutes les balises img avec src="http..." ou src="data:..."
        $pattern = '/<img([^>]*?)src=["\']([^"\']+)["\']([^>]*?)>/i';
        
        return preg_replace_callback($pattern, function ($matches) use ($outputFormat, $quality) {
            $beforeSrc = $matches[1];
            $src = $matches[2];
            $afterSrc = $matches[3];
            
            try {
                // Si c'est deja une image base64
                if (strpos($src, 'data:image') === 0) {
                    // Si on veut convertir le format, convertir meme les base64 existantes
                    if ($outputFormat !== null) {
                        $convertedBase64 = self::convertBase64Image($src, $outputFormat, $quality);
                        return "<img{$beforeSrc}src=\"{$convertedBase64}\"{$afterSrc}>";
                    }
                    // Sinon garder tel quel
                    return $matches[0];
                }
                
                // Decoder les entites HTML dans l'URL (&amp; -> &)
                $src = html_entity_decode($src, ENT_QUOTES | ENT_HTML5);
                
                // Charger l'image une seule fois
                $imageData = @file_get_contents($src);
                if ($imageData === false) {
                    return $matches[0];
                }
                
                if ($outputFormat !== null) {
                    // Creer une image base64 temporaire depuis les donnees
                    $finfo = new \finfo(FILEINFO_MIME_TYPE);
                    $mimeType = $finfo->buffer($imageData);
                    $tempBase64 = "data:{$mimeType};base64," . base64_encode($imageData);
                    
                    // Convertir vers le format souhaite
                    $base64Src = self::convertBase64Image($tempBase64, $outputFormat, $quality);
                } else {
                    // Garder le format original
                    $finfo = new \finfo(FILEINFO_MIME_TYPE);
                    $mimeType = $finfo->buffer($imageData);
                    $base64 = base64_encode($imageData);
                    $base64Src = "data:{$mimeType};base64,{$base64}";
                }
                
                return "<img{$beforeSrc}src=\"{$base64Src}\"{$afterSrc}>";
                
            } catch (\Exception $e) {
                error_log("Error converting image: " . $e->getMessage());
                return $matches[0];
            }
        }, $html);
    }
}
