<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: AiApiController.php,v 1.26.2.7.4.3.2.3 2026/02/03 13:35:09 qvarin Exp $

namespace Pmb\AI\Opac\Controller;

use encoding_normalize;
use Pmb\AI\Library\ParseMarkdown;
use Pmb\AI\Library\Source\{Semantic, SharedList};
use Pmb\AI\Models\AiSessionSemanticModel;
use Pmb\AI\Orm\AiSessionSemanticOrm;
use Pmb\AI\Orm\AISettingsOrm;
use Pmb\Common\Opac\Controller\Controller;

if (stristr($_SERVER['REQUEST_URI'], basename(__FILE__))) {
    die("no access");
}

class AiApiController extends Controller
{
    public const AI_SEMANTIC_TYPE = AiSessionSemanticModel::TYPE_SEMANTIC;

    /**
     * Source
     *
     * @var Semantic|SharedList
     */
    protected $source;

    public function __construct(?object $data = null)
    {
        parent::__construct($data);

        try {
            $aiSettings = AISettingsOrm::fetchActiveAiSettings();
            if (empty($aiSettings)) {
                throw new \Exception('No active AI settings');
            }
        } catch (\Exception $e) {
            $this->ajaxError($e->getMessage(), 403);
        }

        $this->source = new Semantic($aiSettings);
    }

    /**
     * TextGeneration
     *
     * @return void
     */
    public function textGeneration(): void
    {
        if (
            empty($this->data->id) ||
            !is_integer($this->data->id) ||
            !AiSessionSemanticModel::exist($this->data->id, static::AI_SEMANTIC_TYPE)
        ) {
            $this->ajaxError('No id or invalid id', 404);
        }

        $indexList = $_SESSION["ai_search_index_{$this->data->id}_{$this->data->indexQuestion}"] ?? [];
        $_SESSION["ai_search_index_{$this->data->id}_{$this->data->indexQuestion}"] = null;
        session_write_close();

        $aiSessionModel = new AiSessionSemanticModel($this->data->id);
        if (empty($indexList) || !is_array($indexList)) {
            $this->ajaxError('No index list', 404);
        }

        $content = $this->source->textGeneration(
            $aiSessionModel->aiSessionSemantiqueQuestions[$this->data->indexQuestion],
            $indexList
        );
        if (empty($content)) {
            $this->ajaxError('API error (empty content)', 404);
        }

        $parseMarkdown = new ParseMarkdown();
        $content = $parseMarkdown->parseText($content, $aiSessionModel->aiSessionSemantiqueNumObjects[$this->data->indexQuestion]);
        if (!empty($content)) {
            $aiSessionModel->addResponse($this->data->indexQuestion, $content);
        }

        $this->ajaxJsonResponse([
            'result' => $content
        ]);
    }

    /**
     * Send API response
     *
     * @param \CurlResponse|false $response
     * @return void
     */
    protected function sendApiResponse($response): void
    {
        if (false === $response) {
            $this->ajaxError('API error (no response)', 404);
        }
        $data = encoding_normalize::json_decode($response->body);
        $this->ajaxJsonResponse([
            'result' => $data->r ?? ""
        ]);
    }


    /**
     * Permet de renommer une session
     *
     * @return void
     */
    public function SessionRename()
    {
        session_write_close();

        if (empty($this->data->name) || !is_string($this->data->name)) {
            $this->ajaxError('No name or invalid name', 400);
        }

        if (empty($this->data->id) || !is_integer($this->data->id)) {
            $this->ajaxError('No id or invalid id', 400);
        }

        try {
            $setting = new AiSessionSemanticOrm($this->data->id);
            $setting->ai_session_semantique_name = $this->data->name;
            $setting->save();
            $this->ajaxJsonResponse([
                'error' => false,
                'message' => '',
            ]);
        } catch (\Exception $e) {
            $this->ajaxError($e->getMessage());
        }
    }

    /**
     * Permet de supprimer une session
     *
     * @return void
     */
    public function SessionDelete()
    {
        if (empty($this->data->id) || !is_integer($this->data->id)) {
            $this->ajaxError('No id or invalid id', 404);
        }
        try {
            $setting = new AiSessionSemanticOrm($this->data->id);
            $setting->delete();

            for ($i = 0; $i <= $_SESSION["nb_queries"]; $i++) {
                if (
                    !empty($_SESSION["search_type".$i]) &&
                    $_SESSION["search_type".$i] == "ai_search" &&
                    !empty($_SESSION["ai_search_history_{$i}"]) &&
                    $_SESSION["ai_search_history_{$i}"]['ai_session'] == $this->data->id
                ) {
                    unset($_SESSION["user_query".$i], $_SESSION["ai_search_history_{$i}"]);

                }
            }

            $this->ajaxJsonResponse([
                'error' => false,
                'message' => '',
            ]);
        } catch (\Exception $e) {
            $this->ajaxError($e->getMessage());
        }
    }

    /**
     * Renvoie la liste des sessions
     *
     * @return void
     */
    public function sessionList()
    {
        session_write_close();
        $this->ajaxJsonResponse([
            'error' => false,
            'data' => AiSessionSemanticModel::findAll()
        ]);
    }

    /**
     * Retourne une session
     *
     * @param integer $id L'id de la session
     * @return void
     */
    public function session(int $id)
    {
        if (!AiSessionSemanticOrm::exist($id)) {
            $this->ajaxError('No id or invalid id', 404);
        }

        $aiSessionModel = new AiSessionSemanticModel($id);
        $aiSessionModel->fetchHistories();
        $this->ajaxJsonResponse([
            'error' => false,
            'data' => $aiSessionModel
        ]);
    }

    /**
     * Renvoie la dernire session de la base de donnes et la renvoie sous forme de rponse JSON.
     *
     * @return void
     */
    public function sessionLast()
    {
        session_write_close();
        $lastSession = AiSessionSemanticModel::findLast();
        if (null === $lastSession) {
            $this->ajaxError('Last session not found', 404);
        }

        $lastSession->fetchHistories();
        $this->ajaxJsonResponse([
            'error' => false,
            'data' => $lastSession
        ]);
    }

    /**
     * Renvoie l'astuces gnres pour une question
     *
     * @return void
     */
    public function tips()
    {
        session_write_close();

        if (
            empty($this->data->id) ||
            !is_integer($this->data->id) ||
            !AiSessionSemanticModel::exist($this->data->id, static::AI_SEMANTIC_TYPE)
        ) {
            $this->ajaxError('No id or invalid id', 404);
        }

        $aiSessionModel = new AiSessionSemanticModel($this->data->id);
        $response = $this->source->tips(
            $aiSessionModel->aiSessionSemantiqueQuestions[$this->data->indexQuestion],
        );

        if (!empty($response)) {
            $response['conseil'] = ParseMarkdown::parseTips($response['conseil'] ?? "");
            $response['boolean'] = ParseMarkdown::parseBoolean($response['boolean'] ?? "");

            $this->ajaxJsonResponse([
                'error' => false,
                'data' => $response
            ]);
        } else {
            $this->ajaxError('API error (no response)', 404);
        }
    }

}
