<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: EmbeddingsTemplateModel.php,v 1.1.2.1 2026/02/06 09:24:11 qvarin Exp $

namespace Pmb\AI\Models;

use H2o;
use Pmb\AI\Orm\EmbeddingsTemplateOrm;
use Pmb\Common\Models\Model;
use record_datas;

if (stristr($_SERVER['REQUEST_URI'], basename(__FILE__))) {
    die("no access");
}

class EmbeddingsTemplateModel extends Model
{
    public const AVAILABLE_TYPE = [
        TYPE_NOTICE,
        TYPE_EXPLNUM
    ];

    /**
     * Correspond au namespace de l'ORM
     *
     * @var string
     */
    protected $ormName = EmbeddingsTemplateOrm::class;

    /**
     * Type d'entit (see: EmbeddingsTemplateOrm::AVAILABLE_TYPE)
     *
     * @var int
     */
    protected $entityType;

    /**
     * Template
     *
     * @var string
     */
    protected $template = '';

    /**
     * Constructor
     *
     * @param integer $type (see: EmbeddingsTemplateOrm::AVAILABLE_TYPE)
     */
    private function __construct(int $type)
    {
        $this->id = $type;
        $this->entityType = $type;
        if ($type !== TYPE_EXPLNUM) {
            $this->fetchData();
        }
    }

    /**
     * Retourne une instance du model
     *
     * @param integer $type (see: EmbeddingsTemplateOrm::AVAILABLE_TYPE)
     * @return self
     */
    public static function getInstance(int $type): self
    {
        if (!in_array($type, self::AVAILABLE_TYPE, true)) {
            throw new \InvalidArgumentException('Unknown type entity');
        }

        return new self($type);
    }

    /**
     * Definit le template
     *
     * @param string $template
     * @return void
     */
    public function setTemplate(string $template): void
    {
        $this->template = $template;
    }

    /**
     * Retourne le template
     *
     * @return string
     */
    public function getTemplate(): string
    {
        return $this->template;
    }

    /**
     * Retourne le type d'entite
     *
     * @return integer
     */
    public function getType(): int
    {
        return $this->entityType;
    }

    /**
     * Sauvegarde le template
     *
     * @return void
     */
    public function save(): void
    {
        if (TYPE_EXPLNUM === $this->entityType) {
            // On ne fait pas de template pour les exemplaires numeriques
            return;
        }

        if (EmbeddingsTemplateOrm::exist($this->entityType)) {
            $embeddingsTemplateOrm = new EmbeddingsTemplateOrm($this->entityType);
        } else {
            $embeddingsTemplateOrm = new EmbeddingsTemplateOrm();
            $embeddingsTemplateOrm->entity_type = $this->entityType;
        }

        $embeddingsTemplateOrm->template = $this->template;
        $embeddingsTemplateOrm->save();
    }

    /**
     * Recupere tous les templates
     *
     * @return array
     */
    public static function fetchTemplates(): array
    {
        $result = [];
        $embeddingsTemplateCollections = EmbeddingsTemplateOrm::findAll(EmbeddingsTemplateOrm::$idTableName);
        foreach ($embeddingsTemplateCollections as $embeddingsTemplateCollection) {
            $result[$embeddingsTemplateCollection->entity_type] = $embeddingsTemplateCollection->template;
        }
        return $result;
    }

    /**
     * Rendu du template
     *
     * @param int $entityId
     * @return string
     */
    public function render(int $entityId): string
    {
        try {
            switch ($this->entityType) {
                case TYPE_NOTICE:
                    $data = ['record' => record_datas::get_instance($entityId) ];
                    $content = H2o::parseString($this->cleanTemplate($this->template))->render($data);
                    $content = trim($content);
                    break;

                case TYPE_EXPLNUM:
                    $query = "SELECT explnum_index_wew FROM explnum WHERE explnum_id = " . $entityId;
                    $result = pmb_mysql_query($query);

                    if (pmb_mysql_num_rows($result)) {
                        $row = pmb_mysql_fetch_assoc($result);
                        $content = $row['explnum_index_wew'];
                    } else {
                        $content = '';
                    }
                    break;

                default:
                    $content = '';
                    break;
            }
        } catch(\Exception $e) {
            trigger_error($e->getMessage(), E_USER_WARNING);
            $content = '';
        }

        return $content;
    }

    /**
     * Nettoie le template en supprimant les retours  la ligne et les espaces multiples
     *
     * @param string $template
     * @return string
     */
    private function cleanTemplate(string $template): string
    {
        $template = str_replace(["\t", "\n"], '', $template);
        $template = preg_replace('/ +/', ' ', $template);
        return $template;
    }
}
