<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: SharedList.php,v 1.1.2.7.2.2 2025/11/07 15:23:04 gneveu Exp $

namespace Pmb\AI\Library\Source;

if (stristr($_SERVER['REQUEST_URI'], '/'.basename(__FILE__))) {
    die("no access");
}

use emprunteur;
use encoding_normalize;
use Pmb\AI\Library\Chunker\TextChunker;
use Pmb\AI\Models\AiSharedListDocnumModel;
use Pmb\AI\Orm\AiSharedListOrm;

class SharedList extends AbstractSource implements Source
{

    /**
     * Shared List
     *
     * @var int
     */
    protected $sharedListId = 0;

    /**
     * Emprunteur
     *
     * @var null|emprunteur
     */
    private $emprunteur = null;

    /**
     * fetchData
     *
     * @param integer $idSetting
     * @return void
     */
    public function fetchData(int $idSetting)
    {
        $settings = new AiSharedListOrm($idSetting);
        $this->settings = json_decode($settings->settings_ai_shared_list);
        if (null === $this->settings) {
            $this->settings = new \stdClass();
        }
    }

    /**
     * Get Emprunteur
     *
     * @return emprunteur
     */
    private function getEmprunteur(): emprunteur
    {
        if (!isset($this->emprunteur)) {
            $this->emprunteur = new emprunteur($_SESSION['id_empr_session']);
        }
        return $this->emprunteur;
    }

    /**
     * Get Prompt System
     *
     * @return string
     */
    protected function getPromptSystem(): string
    {
        $prompt = $this->settings->prompt->{$this->getEmprunteur()->categ} ?? $this->settings->prompt->{0};
        $prompt = empty($prompt->prompt_system) ? $this->settings->prompt->{0} : $prompt;
        return $prompt->prompt_system ?? '';
    }

    /**
     * Get Prompt User
     *
     * @return string
     */
    protected function getPromptUser(): string
    {
        $prompt = $this->settings->prompt->{$this->getEmprunteur()->categ} ?? $this->settings->prompt->{0};
        $prompt = empty($prompt->prompt_system) ? $this->settings->prompt->{0} : $prompt;
        return $prompt->prompt_user ?? '';
    }

    /**
     * Tips
     *
     * @param string $question
     * @return array
     */
    public function tips(string $question)
    {
        return [];
    }

    /**
     * Set Shared List
     *
     * @param integer $sharedListId
     * @return void
     */
    public function setSharedListId(int $sharedListId)
    {
        $this->sharedListId = $sharedListId;
    }

    /**
     * Indexation
     *
     * @param integer $limit
     * @return array
     */
    public function indexation(int $limit): array
    {
        $data = AiSharedListDocnumModel::getEntityDataAi($this->sharedListId, $limit);

        foreach ($data["entities"] as $entity) {
            if ($entity['object_type'] === TYPE_SHARED_LIST_EXPLNUM) {
                $chunker = new TextChunker();
                $chunks = $chunker->chunk($entity["content"]);

                $embeddings = $this->service->textToEmbeddings(array_column($chunks, 'data'));
                foreach ($chunks as $key => $_) {
                    $chunks[$key]['data'] = $embeddings[$key];
                }

                $aiSharedListDocnumModel = new AiSharedListDocnumModel($entity['object_id']);
                $aiSharedListDocnumModel->flagAiSharedListDocnum = 1;
                $aiSharedListDocnumModel->embeddingsAiSharedListDocnum = encoding_normalize::json_encode($chunks);
                $aiSharedListDocnumModel->save();
            }
        }

        return [
            "count" => $data["count"],
            "countIndexed" => $data["countIndexed"]
        ];
    }

    /**
     * Search
     *
     * @param string $userQuery
     * @return array{object: int, data: array}
     */
    public function search(string $userQuery): array
    {
        $textEmbeddings = $this->service->textToEmbeddings($userQuery);
        if (empty($textEmbeddings)) {
            return [
            	'object' => AbstractSource::OBJECT_RETRY_AFTER,
                'data' => [
                    'retry-after' => 10
                ]
            ];
        }

        $queryEmbeddings = array_shift($textEmbeddings);
        $databaseEmbeddings = $this->getDatabaseEmbeddings();
        $similarEmbeddings = $this->findSimilarAboveThreshold($queryEmbeddings, $databaseEmbeddings, floatval(($this->settings->min_score / 100)));

        $data = [];
        foreach ($similarEmbeddings as $similarEmbedding) {
            $index = $similarEmbedding['object_type'] . '_' . $similarEmbedding['object_id'];
            $data[$index] ??= [
                'object_type' => $similarEmbedding['object_type'],
                'object_id' => $similarEmbedding['object_id'],
                'score' => 0,
                'pertinent_content' => []
            ];

            $data[$index]['score'] = max([$data[$index]['score'], $similarEmbedding['similarity']]);
            $data[$index]['pertinent_content'][] = [
                'offset' => $similarEmbedding['data']['offset'],
                'length' => $similarEmbedding['data']['length']
            ];
        }

        // Tri des resultats par score decroissant
        usort($data, function ($a, $b) {
            if ($a['score'] === $b['score']) {
                return 0;
            }

            return ($a['score'] > $b['score']) ? -1 : 1;
        });

        return [
        	'object' => AbstractSource::OBJECT_RESPONSE,
            'data' => array_values($data)
        ];
    }

    /**
     * Get database embeddings
     *
     * @return array
     */
    protected function getDatabaseEmbeddings(): array
    {

        $embeddings = [];

        $query = "SELECT notice_id FROM notices
      JOIN opac_liste_lecture_notices
                ON notice_id = opac_liste_lecture_notice_num
                AND opac_liste_lecture_num = " . $this->sharedListId . "
            WHERE notices.embeddings IS NOT NULL";

        $result = pmb_mysql_query($query);
        if (pmb_mysql_num_rows($result)) {
            while ($row = pmb_mysql_fetch_assoc($result)) {
                $embeddings[] = [
                    'object_type' => TYPE_NOTICE,
                    'object_id' => $row['notice_id'],
                ];
            }

            pmb_mysql_free_result($result);
        }

        if ($this->getSettings()->indexation_choice->docnum) {
            $query = "SELECT explnum_id FROM explnum
                JOIN notices
                    ON notices.notice_id = explnum.explnum_notice
                JOIN caddie_content
                    ON notices.notice_id = caddie_content.object_id
                JOIN opac_liste_lecture_notices
                    ON notice_id = opac_liste_lecture_notice_num
                    AND opac_liste_lecture_num = " . $this->sharedListId . "
                WHERE explnum.explnum_embeddings IS NOT NULL AND explnum.explnum_embeddings != ''";

            $result = pmb_mysql_query($query);
            if (pmb_mysql_num_rows($result)) {
                while ($row = pmb_mysql_fetch_assoc($result)) {
                    $embeddings[] = [
                        'object_type' => TYPE_EXPLNUM,
                        'object_id' => $row['explnum_id'],
                    ];
                }

                pmb_mysql_free_result($result);
            }
        }

        $query = "SELECT id_ai_shared_list_docnum FROM ai_shared_list_docnum
            WHERE num_list_ai_shared_list_docnum = " . $this->sharedListId . "
                AND embeddings_ai_shared_list_docnum IS NOT NULL AND embeddings_ai_shared_list_docnum != ''";

        $result = pmb_mysql_query($query);
        if (pmb_mysql_num_rows($result)) {
            while ($row = pmb_mysql_fetch_assoc($result)) {
                $embeddings[] = [
                    'object_type' => TYPE_SHARED_LIST_EXPLNUM,
                    'object_id' => $row['id_ai_shared_list_docnum'],
                ];
            }
            pmb_mysql_free_result($result);
        }

        return $embeddings;
    }

}
