<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: Service.php,v 1.1.2.6.2.5 2026/02/13 12:44:17 gneveu Exp $

namespace Pmb\AI\Library\Services;

if (stristr($_SERVER['REQUEST_URI'], '/'.basename(__FILE__))) {
    die("no access");
}

use encoding_normalize;
use Pmb\AI\Library\Parser\AiParserManifest;
use Pmb\AI\Orm\AiServicesConfigOrm;

class Service
{
    /**
     * ID_SERVICE_CONFIG
     *
     * @var int
     */
    public const ID_SERVICE_CONFIG = 1;

    /**
     * parameters
     *
     * @var array
     */
    public $parameters;

    /**
     * @return void
     */
    public function __construct()
    {
        $this->fetchData();
    }

    /**
     *  Initialise les parametres
     *
     * @return void
     */
    public function fetchData()
    {
        if (AiServicesConfigOrm::exist(self::ID_SERVICE_CONFIG)) {
            $aISettingsOrm = new AiServicesConfigOrm(self::ID_SERVICE_CONFIG);
            $servicesConfig = encoding_normalize::json_decode($aISettingsOrm->settings_ai_services_config, true);
	        $this->parameters = $servicesConfig;
        }
    }

    /**
     * Enregistre les parametres
     * @param array $parameters
     *
     * @return bool|string
     */
    public function save(array $parameters)
    {
        try {
            $parameters["id"] = intval($parameters["id"] ?? 0);
            if (empty($parameters["id"]) || $parameters["id"] < 1) {
                return false;
            }

            $agent = self::getService($parameters["id"]);
            if (!$agent) {
                return false;
            }

            try {
                $agent = new $agent['namespace']();
            } catch (\Exception $e) {
                return false;
            }

            if ($agent->checkParameters($parameters) == false) {
                return false;
            }

            if (AiServicesConfigOrm::find("id_ai_services_config", self::ID_SERVICE_CONFIG)) {
                $aiServicesOrm = new AiServicesConfigOrm(self::ID_SERVICE_CONFIG);
            } else {
                $aiServicesOrm = new AiServicesConfigOrm();
            }
            $aiServicesOrm->settings_ai_services_config = encoding_normalize::json_encode($parameters);
            $aiServicesOrm->save();
            return true;
        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }

    /**
     * Recupere les parametres
     *
     * @return array
     */
    public function getParameters()
    {
        return $this->parameters;
    }

    /**
     * Recupere la liste des services
     *
     * @return array
     */
    public static function getServices()
    {
        return AiParserManifest::parseServicesList();
    }

    /**
     * Recupere un service
     *
     * @param int $idService
     * @return array|false
     */
    public static function getService(int $idService)
    {
        $services = self::getServices();
        return $services[$idService] ?? false;
    }

    /**
     * Recupere un agent
     *
     * @return Agent
     */
    public function getAgent()
    {
        $agent = self::getService($this->parameters["id"]);
        $instance = new $agent['namespace']();
        $instance->setParameters($this->parameters);
        return $instance;
    }

    /**
     * Recupere les embeddings
     *
     * @param string|array $text
     * @return array
     */
    public function textToEmbeddings(string|array $text)
    {
        return $this->getAgent()->textToEmbeddings($text);
    }

    /**
     * Recupere les suggestions
     *
     * @param string $promptSystem
     * @param string $promptUser
     * @return array
     */
    public function tips(string $promptSystem, string $promptUser)
    {
        $response = $this->getAgent()->chat($promptSystem, $promptUser, true);
        if (!empty($response)) {
            $response = encoding_normalize::json_decode($response, true);
            return $response ?? [];
        }
        return [];
    }

    /**
     * Valide le token
     *
     * @param int $serviceId
     * @param array $parameters
     * @return bool
     */
    public function validateToken(int $serviceId, array $parameters = []): bool
    {
        $service = self::getService($serviceId);
        if (!$service) {
            return false;
        }

        try {
            $agent = new $service['namespace']();
            if (!method_exists($agent, 'validateToken')) {
                return false;
            }
            return call_user_func_array([$agent, 'validateToken'], [$parameters]);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Recupere les modeles
     *
     * @param int $serviceId
     * @param array $parameters
     * @return bool|array
     */
    public function getModels(int $serviceId, array $parameters = [])
    {
        $service = self::getService($serviceId);
        if (!$service) {
            return false;
        }

        try {
            $agent = new $service['namespace']();
            if (!method_exists($agent, 'getModels')) {
                return false;
            }
            return $agent->getModels($parameters);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Chat
     *
     * @param string $promptSystem
     * @param string $promptUser
     * @return string
     */
    public function chat(string $promptSystem, string $promptUser): string
    {
        return $this->getAgent()->chat($promptSystem, $promptUser);
    }

}
