<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: Ollama.php,v 1.1.2.4.2.3 2025/09/16 08:15:15 qvarin Exp $

namespace Pmb\AI\Library\Services\Ollama;

if (stristr($_SERVER['REQUEST_URI'], '/'.basename(__FILE__))) {
    die("no access");
}

use encoding_normalize;
use Pmb\AI\Library\Services\Agent;
use Pmb\AI\Library\Services\AgentUtils;

class Ollama implements Agent
{
    use AgentUtils;

    /**
     * URI utilisee pour la conversation
     */
    public const URI_COMPLETION = "/api/chat";

    /**
     * URI utilisee pour le calcul des embeddings
     */
    public const URI_EMBEDDINGS = "/api/embed";

    /**
     * URI utilisee pour la recuperation des models
     */
    public const URI_MODELS = "/api/tags";

    /**
     * Parameters
     */
    private $parameters;

    /**
     * Set parameters
     *
     * @param array $parameters
     * @return void
     */
    public function setParameters(array $parameters): void
    {
        $this->parameters = $parameters;
    }

    /**
     * Get parameters
     *
     * @return array
     */
    public function getParameters(): array
    {
        return $this->parameters;
    }

    /**
     * Convert text to embeddings
     *
     * @param string|array  $text
     * @return array
     */
    public function textToEmbeddings(string|array  $text): array
    {
        $embeddingSend = [
            "model" => $this->parameters['embeddings_model'],
            "input" => $text,
        ];

        $response = $this->post(
            rtrim($this->parameters['url_server'], '/') . self::URI_EMBEDDINGS,
            $this->headers(),
            encoding_normalize::json_encode($embeddingSend)
        );
        // TODO : Revoir le strpos de 200 OK souci dans l'application Ollama qui retourne ([Status] => 100 ContinueHTTP/1.1 200 OK)
        if (($response instanceof \CurlResponse) && strpos(strtolower($response->headers['Status']), '200') !== false) {
            $response = encoding_normalize::json_decode($response->body, true);
            return $response['embeddings'];
        }
        return [];
    }

    /**
     * Chat
     *
     * @param string $promptSystem
     * @param string $promptUser
     * @return string
     */
    public function chat(string $promptSystem, string $promptUser): string
    {
        $data = [
            "model" => $this->parameters['chat_model'],
            "messages" => [
                [ "role" => "system", "content" => $promptSystem ],
                [ "role" => "user", "content" => $promptUser ]
            ],
            "stream" => false
        ];

        $options =  $this->options();
        if (!empty($options)) {
            $data['options'] = $options;
        }

        $response = $this->post(
            rtrim($this->parameters['url_server'], '/') . self::URI_COMPLETION,
            $this->headers(),
            encoding_normalize::json_encode($data)
        );

        // TODO : Revoir le strpos de 200 OK souci dans l'application Ollama qui retourne ([Status] => 100 ContinueHTTP/1.1 200 OK)
        if (($response instanceof \CurlResponse) && strpos(strtolower($response->headers['Status']), '200') !== false) {
            $response = encoding_normalize::json_decode($response->body, true);
            return $response["message"]["content"];
        }
        return "";
    }

    /**
     * Permet de recuperer la liste des models
     *
     * @param array $parameters
     * @return array
     */
    public function getModels(array $parameters): array
    {
        $listModels = [];
        $headers['Authorization'] = 'Bearer ' . $parameters['auth_token'] ?? '';
        $response = $this->get(rtrim($parameters['url_server'], '/') . self::URI_MODELS, $headers);
        if (($response instanceof \CurlResponse) &&  $response->headers['Status-Code'] == 200) {
            $body = encoding_normalize::json_decode($response->body, true);
            foreach ($body["models"] as $model) {
                if (!in_array($model["name"], $listModels)) {
                    $listModels[] = $model["name"];
                }
            }
        }

        return $listModels;
    }

    /**
     * Vrification des paramtres
     *
     * @param array $parameters
     * @return boolean
     */
    public function checkParameters(array $parameters): bool
    {
        if (isset($parameters['options']['num_ctx']) && !(is_numeric($parameters['options']['num_ctx']) || $parameters['options']['num_ctx'] === "")) {
            return false;
        }

        if (isset($parameters['options']['temperature']) && !(is_numeric($parameters['options']['temperature']) || $parameters['options']['temperature'] === "")) {
            return false;
        }

        if (isset($parameters['options']['top_p']) && !(is_numeric($parameters['options']['top_p']) || $parameters['options']['top_p'] === "")) {
            return false;
        }

        if (isset($parameters['options']['top_k']) && !(is_numeric($parameters['options']['top_k']) || $parameters['options']['top_k'] === "")) {
            return false;
        }

        if (isset($parameters['url_server']) && isset($parameters['chat_model']) && isset($parameters['embeddings_model'])) {
            $ollamaParams = [
                'url_server' => $parameters['url_server'],
                'auth_token' => $parameters['auth_token'] ?? '',
            ];
            $models = $this->getModels($ollamaParams);
            if (
                in_array($parameters['chat_model'], $models, true) &&
                in_array($parameters['embeddings_model'], $models, true)
            ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get headers
     *
     * @return array
     */
    private function headers(): array
    {
        $headers = [];
        if (!empty($this->parameters['auth_token'])) {
            $headers['Authorization'] = 'Bearer ' . $this->parameters['auth_token'] ?? '';
        }
        return $headers;
    }

    /**
     * Get options
     *
     * @return array
     */
    private function options(): array
    {
        $options = [];

        if (isset($this->parameters['options']['num_ctx']) && is_numeric($this->parameters['options']['num_ctx'])) {
            $options['num_ctx'] = $this->parameters['options']['num_ctx'];
        }

        if (isset($this->parameters['options']['temperature']) && is_numeric($this->parameters['options']['temperature'])) {
            $options['temperature'] = $this->parameters['options']['temperature'];
        }

        if (isset($this->parameters['options']['top_p']) && is_numeric($this->parameters['options']['top_p'])) {
            $options['top_p'] = $this->parameters['options']['top_p'];
        }

        if (isset($this->parameters['options']['top_k']) && is_numeric($this->parameters['options']['top_k'])) {
            $options['top_k'] = $this->parameters['options']['top_k'];
        }

        return $options;
    }
}