<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: RetryManager.php,v 1.1.2.2 2025/11/07 15:23:04 gneveu Exp $

namespace Pmb\AI\Library;

if (stristr($_SERVER['REQUEST_URI'], '/'.basename(__FILE__))) {
    die("no access");
}

/**
 * RetryManager - Gre les mcanismes de retry pour les appels API AI
 *
 * Gre les tentatives de reconnexion quand l'API ne rpond pas,
 * avec timeout global et limite de retries pour viter les boucles infinies.
 */
class RetryManager
{
    /**
     * Nombre maximum de retries sans rponse valide de l'API
     */
    public const MAX_RETRIES = 3;

    /**
     * Dlai de retry en secondes (fixe, pas de backoff)
     */
    public const RETRY_DELAY = 10;

    /**
     * Timeout global en secondes (temps max total pour toutes les tentatives)
     */
    public const TIMEOUT_GLOBAL = 60;

    /**
     * Cl de session pour stocker les infos de retry
     */
    private const SESSION_KEY_PREFIX = 'ai_retry_';

    /**
     * Cl pour le timestamp de dbut
     */
    private const SESSION_START_TIME = 'start_time';

    /**
     * Cl pour le compteur de retries
     */
    private const SESSION_RETRY_COUNT = 'count';

    /**
     * Cl pour le dernier timestamp de retry
     */
    private const SESSION_LAST_ATTEMPT = 'last_attempt';

    /**
     * Identifiant unique de la session de retry
     *
     * @var string
     */
    private string $sessionId;

    /**
     * Constructeur
     *
     * @param string $sessionId Identifiant unique pour cette session de retry (ex: ai_session_123)
     */
    public function __construct(string $sessionId)
    {
        $this->sessionId = $sessionId;
    }

    /**
     * Initie une nouvelle session de retry
     *
     * Appele lors du premier appel API
     *
     * @return void
     */
    public function initSession(): void
    {
        $sessionKey = $this->getSessionKey();
        $_SESSION[$sessionKey] = [
            self::SESSION_START_TIME => time(),
            self::SESSION_RETRY_COUNT => 0,
            self::SESSION_LAST_ATTEMPT => time()
        ];
    }

    /**
     * Enregistre une tentative choue
     *
     * @return void
     */
    public function recordFailedAttempt(): void
    {
        $sessionKey = $this->getSessionKey();

        if (!isset($_SESSION[$sessionKey])) {
            $this->initSession();
        }

        $_SESSION[$sessionKey][self::SESSION_RETRY_COUNT]++;
        $_SESSION[$sessionKey][self::SESSION_LAST_ATTEMPT] = time();
    }

    /**
     * Vrifie si on peut faire un retry
     *
     * Retourne vrai si:
     * - On n'a pas atteint le nombre max de retries
     * - On n'a pas dpass le timeout global
     *
     * @return bool true si un retry est possible, false sinon
     */
    public function canRetry(): bool
    {
        $sessionKey = $this->getSessionKey();

        // Pas de session = premire tentative
        if (!isset($_SESSION[$sessionKey])) {
            $this->initSession();
            return true;
        }

        $session = $_SESSION[$sessionKey];
        $retryCount = $session[self::SESSION_RETRY_COUNT] ?? 0;
        $startTime = $session[self::SESSION_START_TIME] ?? time();

        // Vrifier le nombre de retries
        if ($retryCount >= self::MAX_RETRIES) {
            return false;
        }

        // Vrifier le timeout global
        $elapsedTime = time() - $startTime;
        if ($elapsedTime >= self::TIMEOUT_GLOBAL) {
            return false;
        }

        return true;
    }

    /**
     * Obtient le dlai de retry en secondes
     *
     * Retourne le nombre de secondes avant le prochain retry
     *
     * @return int Dlai en secondes
     */
    public function getRetryDelay(): int
    {
        return self::RETRY_DELAY;
    }

    /**
     * Obtient le nombre de retries effectus
     *
     * @return int Nombre de tentatives choues
     */
    public function getRetryCount(): int
    {
        $sessionKey = $this->getSessionKey();

        if (!isset($_SESSION[$sessionKey])) {
            return 0;
        }

        return $_SESSION[$sessionKey][self::SESSION_RETRY_COUNT] ?? 0;
    }

    /**
     * Obtient le temps coul depuis le dbut des tentatives (en secondes)
     *
     * @return int Temps en secondes
     */
    public function getElapsedTime(): int
    {
        $sessionKey = $this->getSessionKey();

        if (!isset($_SESSION[$sessionKey])) {
            return 0;
        }

        $startTime = $_SESSION[$sessionKey][self::SESSION_START_TIME] ?? time();
        return time() - $startTime;
    }

    /**
     * Obtient le temps restant avant le timeout global (en secondes)
     *
     * @return int Temps en secondes (0 ou moins si dpass)
     */
    public function getRemainingTime(): int
    {
        $elapsedTime = $this->getElapsedTime();
        return self::TIMEOUT_GLOBAL - $elapsedTime;
    }

    /**
     * Nettoie la session de retry
     *
     * Appele aprs une rponse valide de l'API ou un timeout
     *
     * @return void
     */
    public function cleanSession(): void
    {
        $sessionKey = $this->getSessionKey();
        unset($_SESSION[$sessionKey]);
    }

    /**
     * Obtient un rsum des tentatives de retry
     *
     * @return array Rsum avec 'count', 'elapsed_time', 'remaining_time'
     */
    public function getSummary(): array
    {
        return [
            'retry_count' => $this->getRetryCount(),
            'elapsed_time' => $this->getElapsedTime(),
            'remaining_time' => $this->getRemainingTime(),
            'max_retries' => self::MAX_RETRIES,
            'timeout_global' => self::TIMEOUT_GLOBAL,
            'retry_delay' => self::RETRY_DELAY
        ];
    }

    /**
     * Gnre la cl de session unique pour cette session de retry
     *
     * @return string
     */
    private function getSessionKey(): string
    {
        return self::SESSION_KEY_PREFIX . $this->sessionId;
    }
}
