<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: Cluster.php,v 1.1.2.3 2025/04/11 08:58:47 qvarin Exp $

namespace Pmb\AI\Library\Clusters;

if (stristr($_SERVER['REQUEST_URI'], '/'.basename(__FILE__))) {
    die("no access");
}

use Exception;
use InvalidArgumentException;
use Pmb\AI\Library\Utils;

class Cluster
{
    public const MAX_STAGNATION = 5;
    public const PERCENT_STAGNATION = 1 / 100;

    /**
     * Le nombre de fois que le cluster est stagnant
     *
     * @var integer
     */
    private $stagnationCount = 0;

    /**
     * L'ancien cosine difference entre le nouvel embeddings et l'embeddings du cluster
     *
     * @var float
     */
    private $cosinedifference = 0;

    /**
     * L'embeddings moyen
     *
     * @var array<int, float>
     */
    private $embeddingsMeans = [];

    /**
     * L'embeddings
     *
     * @var array<int, float>
     */
    private $embeddings = [];

    /**
     * L'id du cluster
     *
     * @var int
     */
    private $id;

    /**
     * Les objets du cluster
     *
     * @var array
     */
    private $objects = [];

    /**
     * Creer un cluster
     *
     * @param integer $objectType
     * @param integer $objectId
     * @return Cluster
     */
    public static function create(int $objectType, int $objectId): Cluster
    {
        $embeddings = self::fetchEmbeddings($objectType, $objectId);
        $embeddings = Utils::calculateAverageEmbeddings($embeddings);

        $cluster = new self();
        $cluster->embeddings = $embeddings;
        $cluster->initEmbeddingsMeans();
        $cluster->save($cluster->embeddingsMeans);

        return $cluster;
    }

    /**
     * Enregistrer le cluster
     *
     * @param array|null $embeddings
     * @return void
     */
    public function save(?array $embeddings = null): void
    {
        $embeddings ??= $this->embeddings;

        if ($this->id) {
            $query = 'UPDATE clusters SET embeddings = "' . json_encode($embeddings) . '" WHERE id = ' . $this->id;
            pmb_mysql_query($query);
        } else {
            $query = 'INSERT INTO clusters (embeddings) VALUES ("' . json_encode($embeddings) . '")';
            pmb_mysql_query($query);
            $this->id = intval(pmb_mysql_insert_id());
        }

        if (!empty($this->objects)) {
            $this->saveObjects();
        }
    }

    /**
     * Enregistrer les objets du cluster
     *
     * @return void
     */
    private function saveObjects(): void
    {
        $query = 'INSERT INTO cluster_contents
                (num_cluster, num_object, type_object)
            VALUES ';

        $values = [];
        foreach ($this->objects as $object) {
            $values[] = '(' . $this->id . ', ' . $object['type'] . ', ' . $object['id'] . ')';
        }

        $query .= implode(', ', $values);
        pmb_mysql_query($query);
    }

    /**
     * Ajouter un objet au cluster
     *
     * @param integer $type
     * @param integer $id
     * @return void
     */
    public function addObject(int $type, int $id): void
    {
        $this->objects[] = ['type' => $type, 'id' => $id];
    }

    /**
     * Indique si le cluster est stagnant
     *
     * @return boolean
     */
    public function isStagnant(): bool
    {
        return $this->stagnationCount >= Cluster::MAX_STAGNATION;
    }

    /**
     * Mettre  jour l'embeddings moyen
     *
     * @param array $embeddings
     * @return void
     */
    public function updateEmbeddingsMean(array &$embeddings)
    {
        for ($c = 0, $lenChunks = count($embeddings); $c < $lenChunks; $c++) {
            for ($e = 0, $lenEmbeddings = count($embeddings[$c]); $e < $lenEmbeddings; $e++) {
                $this->embeddingsMeans[$e] += $embeddings[$c][$e];
            }
        }
    }

    /**
     * Mettre  jour l'embeddings par rapport aux embeddings moyens
     *
     * @return void
     */
    public function updateEmbeddings(): void
    {
        if ($this->isStagnant()) {
            // Le cluster est stagnant, on ne le plus  jour
            return;
        }

        $newEmbeddings = Utils::computeNormalizeEmbeddings($this->embeddingsMeans);
        if ($this->cosinedifference === 0) {
            // C'est la premire fois qu'on met  jour l'embeddings moyen
            $this->cosinedifference = floatval(1);
            $this->embeddings = $newEmbeddings;
            return;
        }

        $newDifference = Utils::cosineSimilarity($newEmbeddings, $this->embeddings);
        if (abs($this->cosinedifference - $newDifference) > Cluster::PERCENT_STAGNATION) {
            // L'cart entre le nouvelle embeddings et l'embeddings moyens est trop grand (gros dcalage)
            // Donc on met  jour l'embeddings moyen
            $this->cosinedifference = $newDifference;
            $this->stagnationCount = 0;
            $this->embeddings = $newEmbeddings;
        } else {
            $this->stagnationCount++;
        }
    }

    /**
     * Initialiser la moyenne des embeddings
     *
     * @return void
     */
    public function initEmbeddingsMeans(): void
    {
        $this->embeddingsMeans = array_fill(0, count($this->embeddings), 0);
    }

    /**
     * Recuperer les embeddings d'un objet
     *
     * @param integer $type
     * @param integer $id
     * @return array
     * @throws Exception
     * @throws InvalidArgumentException
     */
    private static function fetchEmbeddings(int $type, int $id): array
    {
        switch ($type) {
            case TYPE_NOTICE:
                $query = 'SELECT JSON_EXTRACT(embeddings, "$.data") AS embeddings FROM notices WHERE notice_id = ' . $id;
                break;

            case TYPE_EXPLNUM:
                $query = 'SELECT explnum_embeddings FROM explnum WHERE explnum_id = ' . $id;
                break;

            default:
                throw new InvalidArgumentException('Unsupported object type.');
        }

        $result = pmb_mysql_query($query);
        if (pmb_mysql_num_rows($result)) {
            $row = pmb_mysql_result($result, 0, 0);
            $row = json_decode($row, true);
            pmb_mysql_free_result($result);

            switch ($type) {
                case TYPE_NOTICE:
                    return [$row];

                case TYPE_EXPLNUM:
                    return array_column($row, 'data');
            }
        }

        throw new Exception('Embeddings not found.');
    }

    /**
     * Retourne l'embeddings
     *
     * @return array<int, float>
     */
    public function getEmbeddings(): array
    {
        return $this->embeddings;
    }

    /**
     * Supprimer un objet
     *
     * @param integer $type
     * @param integer $id
     * @return void
     */
    public static function removeObject(int $type, int $id): void
    {
        $query = "SELECT 1 FROM cluster_contents WHERE num_object = " . $id . " AND type_object = " . $type;
        $result = pmb_mysql_query($query);

        if (!pmb_mysql_num_rows($result)) {
            $query = "DELETE FROM cluster_contents WHERE num_object = " . $id . " AND type_object = " . $type;
            pmb_mysql_query($query);
        }
    }
}
