<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: DiffusionHistory.php,v 1.38.2.4.2.8 2026/01/27 09:36:40 jparis Exp $
namespace Pmb\DSI\Models;

use Pmb\Common\Models\Model;
use Pmb\DSI\Helper\LookupHelper;
use Pmb\DSI\Helper\SubscriberHelper;
use Pmb\DSI\Models\CRUD;
use Pmb\Common\Helper\Helper;
use Pmb\DSI\Orm\ContentHistoryOrm;
use Pmb\DSI\Orm\ContentBufferOrm;
use Pmb\Common\Helper\GlobalContext;
use Pmb\DSI\Models\Channel\RootChannel;
use Pmb\DSI\Models\SubscriberList\RootSubscriberList;
use Pmb\DSI\Models\Item\RootItem;
use Pmb\DSI\Models\View\RootView;
use Pmb\DSI\Models\Item\SimpleItem;
use Pmb\Common\Helper\HTML;
use Pmb\DSI\Models\View\WYSIWYGView\WYSIWYGView;
use Pmb\DSI\Orm\SendQueueOrm;
use Pmb\DSI\Orm\DiffusionHistoryOrm;

class DiffusionHistory extends Model implements CRUD
{

	public static $instances = array();

	/**
	 * Diffusion a initialiser (Dans le cas d'un declencheur manuel)
	 * Aucune donnee n'est enregistrer
	 */
	public const INITIALIZED = 0;

	/**
	 * Diffusion a valider, les donnee sont enregistrees
	 */
	public const TO_VALIDATE = 1;

	/**
	 * Diffusion valider (modifie au besoin), pret a l'envoi
	 */
	public const VALIDATED = 2;

	/**
	 * Diffusion envoyee
	 */
	public const SENT = 3;

	/**
	 * Diffusion envoyee
	 */
	public const RESET = 4;

	/**
	 * Diffusion sans donnee dans l'historique
	 */
	public const NODATA = 5;

	public const ITEM_SAVE_IDS = true;

	protected $ormName = "Pmb\DSI\Orm\DiffusionHistoryOrm";

	public $idDiffusionHistory = 0;

	public $numDiffusion = 0;

	public $diffusion = null;

	public $date = null;

	public $formatedDate = null;

	public $totalRecipients = 0;

	public $state = 0;

	public $contentHistory = [];

	public $contentBuffer = [];

	public function __construct(int $id = 0, ?Diffusion $diffusion = null)
	{
		$this->id = $id;

		if (! empty($diffusion)) {
			$this->setDiffusion($diffusion);
		}

		$this->read();
	}

	public function read()
	{
		$this->fetchData();
		$this->fetchRelations();
		$this->formatedDate = $this->getFormatedDate();
	}

	/**
	 * Pour la creation, voir la fonction init()
	 *
	 * @return void
	 */
	public function create($buffer = true)
	{
		$orm = new $this->ormName();
		$orm->num_diffusion = $this->numDiffusion;
		$orm->date = $this->date;
		$orm->total_recipients = $this->totalRecipients;
		$orm->state = $this->state;

		$orm->save();
		$this->id = $orm->{$this->ormName::$idTableName};
		$this->{Helper::camelize($this->ormName::$idTableName)} = $orm->{$this->ormName::$idTableName};
		$this->saveContentBuffer();
	}

	/**
	 * Pour l'update, voir les fonctions toValidate(), validate(), sent()
	 *
	 * @return void
	 */
	public function update()
	{
		$orm = new $this->ormName($this->id);
		$orm->num_diffusion = $this->numDiffusion;
		$orm->date = $this->date;
		$orm->total_recipients = $this->totalRecipients;
		$orm->state = $this->state;

		$orm->save();
		$this->saveContentBuffer();
	}

	public function delete()
	{
		$this->cleanBuffer();

		$this->fetchContents();
		if (! empty($this->contentHistory)) {
			foreach ($this->contentHistory as $contentHistoryList) {
				array_walk($contentHistoryList, function ($contentHistory) {
					$contentHistory->delete();
				});
			}
		}

		$orm = new $this->ormName($this->id);
		$orm->delete();
	}

	protected function fetchRelations()
	{
		$this->fetchDiffusion();
		$this->fetchContents();
		$this->fetchContentBuffer();
	}

	private function fetchDiffusion()
	{
		if ((! isset($this->diffusion) && $this->numDiffusion) || (isset($this->diffusion) && $this->diffusion->id != $this->numDiffusion)) {
			$this->diffusion = Diffusion::getInstance($this->numDiffusion);
		}
	}

	private function fetchContents()
	{
		$this->contentHistory = [];

		$contentHistoryOrm = ContentHistoryOrm::find("num_diffusion_history", $this->id);
		while (count($contentHistoryOrm) > 0) {
			$history = array_shift($contentHistoryOrm);
			$this->contentHistory[$history->type][] = new ContentHistory($history->id_content_history);
		}
	}

	private function fetchContentBuffer()
	{
		$this->contentBuffer = [];

		$contentBufferOrm = ContentBufferOrm::find("num_diffusion_history", $this->id);
		while (count($contentBufferOrm) > 0) {
			$buffer = array_shift($contentBufferOrm);
			$this->contentBuffer[$buffer->type][] = new ContentBuffer($buffer->id_content_buffer);
		}
		$this->setupItemsBuffer();
		$this->setupViewsBuffer();
	}

	/**
	 * La diffusion a bien ete diffuser
	 * On vide le buffer et on remplit notre historique par le contenu du buffer si il est active
	 *
	 * @return void
	 */
	public function saveContentHistory()
	{
		$this->clearBuffer();

		foreach ($this->contentBuffer as $contentBuffer) {
			array_walk($contentBuffer, function ($contentBuffer) {
				$contentHistory = new ContentHistory();
				$contentHistory->id = 0;
				$contentHistory->type = $contentBuffer->type;
				$contentHistory->content = $contentBuffer->content;
				$contentHistory->numDiffusionHistory = $this->id;
				$contentHistory->create();
			});
		}
	}

	public function saveContentBuffer()
	{
		$this->clearBuffer();
		foreach ($this->contentBuffer as $contentBuffer) {
			foreach($contentBuffer as $buffer) {
				$buffer->id = 0;
				$buffer->numDiffusionHistory = $this->id;
				if($buffer->type == ContentBuffer::CONTENT_TYPES_VIEW) {
					// if(! empty($buffer->content["childs"])) {
					// 	$buffer->content["childs"] = [];
					// }
				}
				$buffer->create();
			}
		}
	}

	public function getFormatedDate()
	{
		if (empty($this->date)) {
			return "";
		}

		if (! $this->date instanceof \DateTime) {
			$date = new \DateTime($this->date);
		} else {
			$date = $this->date;
		}

		return $date->format(GlobalContext::msg('dsi_format_date'));
	}

	/**
	 * Creer une DiffusionHistory pour les envois manuels
	 * Aucune donnees n'est sauvegarder
	 *
	 * @return void
	 */
	public function init()
	{
		$this->updateDate();
		$this->state = DiffusionHistory::INITIALIZED;
		$this->create(false);
	}

	/**
	 * Passe la DiffusionHistory a "a valider", puis on fait les sauvegardes
	 *
	 * @return void
	 */
	public function toValidate()
	{
		$this->updateDate();
		$this->state = DiffusionHistory::TO_VALIDATE;

		$this->diffusion->fetchView();
		$this->diffusion->fetchItem();
		$this->diffusion->fetchChannel();
		$this->diffusion->fetchSubscriberList();
		$this->diffusion->fetchEvents();
		$this->diffusion->fetchDiffusionDescriptors();

		$subscriberList = $this->diffusion->getSubscribers();
		$subscribers = RootSubscriberList::getSubscriberListToSend($subscriberList, $this->diffusion->channel::CHANNEL_REQUIREMENTS["subscribers"]) ?? [];

		$this->totalRecipients = count($subscribers) ?? 0;

		$this->contentBuffer = [];

		$this->addContentItem($this->diffusion->item);
		$this->addContentSubscriberList($subscriberList);
		$this->addContentView($this->diffusion->view);
		$this->addContentChannel($this->diffusion->channel);

		if (! $this->id) {
			$this->create();
		} else {
			$this->update();
		}
	}

	/**
	 * Remet tout a zero via la diffusion donnee
	 *
	 * @return void
	 */
	public function reset()
	{
		if ($this->state != DiffusionHistory::TO_VALIDATE) {
			// On doit etre en statut "a valider" pour faire un reset des donnees
			throw new \Exception("Invalid state change to state " . DiffusionHistory::TO_VALIDATE);
		}

		$this->toValidate();
	}

	/**
	 * Passe la DiffusionHistory a "valider" (pret a l'envoi)
	 *
	 * @param Diffusion $diffusion
	 * @return void
	 */
	public function validate()
	{
		if (DiffusionHistory::TO_VALIDATE != $this->state) {
			throw new \Exception("Invalid state change to state " . DiffusionHistory::TO_VALIDATE);
		}

		$this->updateDate();
		$this->state = DiffusionHistory::VALIDATED;
		$this->update();
	}

	/**
	 * DiffusionHistory envoye, ont fait le menage
	 *
	 * @param Diffusion $diffusion
	 */
	public function sent()
	{
		global $dsi_send_automatically;

		if (DiffusionHistory::VALIDATED != $this->state) {
			throw new \Exception("Invalid state change to state " . DiffusionHistory::VALIDATED);
		}

		$diffusionHistoryParser = new DiffusionHistoryParser();
		$item = $diffusionHistoryParser->parseItem($this->contentBuffer[ContentHistory::CONTENT_TYPES_ITEM]);
		$view = $diffusionHistoryParser->parseView($this->contentBuffer[ContentHistory::CONTENT_TYPES_VIEW]);
		$subscribers = $diffusionHistoryParser->parseSubscribers($this->contentBuffer[ContentHistory::CONTENT_TYPES_SUBSCRIBER]);
		$channel = $diffusionHistoryParser->parseChannel($this->contentBuffer[ContentHistory::CONTENT_TYPES_CHANNEL]);

		//On reset les donnees qui vont bien
		$this->diffusion->item = $item;
		$this->diffusion->view = $view;
		$this->diffusion->subscriberList = $subscribers;
		$this->diffusion->channel = $channel;

		$sent = true;
		$render = $view->render($item, $this->diffusion->id, $view->settings->limit ?? 0, "DiffusionPending");
		if ($channel->sendManually()) {

			if (empty($dsi_send_automatically)) {
				$sendQueueOrm = SendQueueOrm::find("num_diffusion_history", $this->id);
				if (empty($sendQueueOrm)) {

					$emprIds = [];
					$subscriberIds = [];
					foreach ($subscribers as $subscriber) {
						if (!empty($subscriber->id)) {
							$subscriberIds[] = $subscriber->id;
							continue;
						}

						$emprIds[] = $subscriber->settings->idEmpr;
					}

					SendQueue::fillQueue($subscriberIds, $emprIds, $this->id, $channel->type);
				}

				$nextQueueElements = SendQueue::getNext($this->id);

				$nextSubscribers = [];
				foreach ($nextQueueElements as $element) {
					foreach ($subscribers as $subscriber) {

						if (
							$element->numSubscriberDiffusion != 0 && isset($subscriber->id) &&
							$subscriber->id == $element->numSubscriberDiffusion
						) {
							$nextSubscribers[] = $subscriber;
							continue;
						}

						if (
							$element->numSubscriberEmpr != 0 && isset($subscriber->settings->idEmpr) &&
							$subscriber->settings->idEmpr == $element->numSubscriberEmpr
						) {
							$nextSubscribers[] = $subscriber;
						}
					}
				}

				$this->diffusion->currentHistory = $this;
			} else {
				$nextSubscribers = $subscribers;
			}


			$channel->setTitle($this->diffusion->name);
			$sent = $channel->send($nextSubscribers, $render, $this->diffusion);

			if (!$sent) {
				return false;
			}

			if (empty($dsi_send_automatically)) {
				SendQueue::flagNext($this->id);
				SendQueue::cleanQueue($this->id);

				$remainingElements = SendQueue::getRemaining($this->id);
				if (!empty($remainingElements)) {
					return true;
				}
			}
		}
		//Pour conserver les donnees issues des lookups H2o dans le cas des DSI privees
		//Peut mieux faire mais ca fait le taff
		if (is_string($render)) {
			if (isset($this->diffusion->settings->isPrivate) && $this->diffusion->settings->isPrivate) {
				$render = LookupHelper::format(SubscriberHelper::format($render, $subscribers[0], false, $this->diffusion), $this->diffusion);
			} else {
				$render = LookupHelper::format($render, $this->diffusion, false);
			}
		}

		$this->addContentRenderView($render);
		$this->cleanContentItem();
		$this->cleanContentSubscriberList();

		// On update le channel pour les stats
		$this->cleanContentChannel();
		$this->addContentChannel($this->diffusion->channel);

		$this->updateDate();
		$this->state = DiffusionHistory::SENT;

		$this->update();

		// On enregistre l'historique
		$this->saveContentHistory();
		$this->diffusion->checkCountHistorySaved();

		return true;
	}

	public function addContentItem(RootItem $item)
	{
		if (!($item instanceof RootItem)) {
			return;
		}

		$childs = [];
		if (! empty($item->childs)) {
			$childs = $item->childs;
			$item->childs = [];
		}

		SimpleItem::$ignoreResultsToArray = false;
		$content = Helper::toArray($item, "");
		SimpleItem::$ignoreResultsToArray = true;

		$content['removed'] = [];

		$contentBuffer = new ContentBuffer();
		$contentBuffer->setContent($content);
		$contentBuffer->type = ContentBuffer::CONTENT_TYPES_ITEM;
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_ITEM][] = $contentBuffer;

		foreach ($childs as $itemChild) {
			$this->addContentItem($itemChild);
		}
	}

	public function addContentSubscriberList($subscriberList)
	{
		$subscribers = RootSubscriberList::getSubscriberListToSend($subscriberList, $this->diffusion->channel::CHANNEL_REQUIREMENTS["subscribers"]);
		$contentBuffer = new ContentBuffer();
		$contentBuffer->setContent(Helper::toArray($subscribers));
		$contentBuffer->type = ContentBuffer::CONTENT_TYPES_SUBSCRIBER;
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_SUBSCRIBER][] = $contentBuffer;
	}

	public function addContentChannel(RootChannel $channel)
	{
		if (!($channel instanceof RootChannel)) {
			return;
		}

		$contentBuffer = new ContentBuffer();
		$contentBuffer->setContent(Helper::toArray($channel));
		$contentBuffer->type = ContentBuffer::CONTENT_TYPES_CHANNEL;
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_CHANNEL][] = $contentBuffer;
	}

	public function addContentView(RootView $view)
	{
		if (!($view instanceof RootView)) {
			return;
		}

		$childs = [];
		if (! empty($view->childs)) {
			$childs = $view->childs;
			$view->childs = [];
		}

		$contentBuffer = new ContentBuffer();
		$contentBuffer->setContent(Helper::toArray($view, ""));
		$contentBuffer->type = ContentBuffer::CONTENT_TYPES_VIEW;
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_VIEW][] = $contentBuffer;

		foreach ($childs as $viewChild) {
			$this->addContentView($viewChild);
		}
	}

	public function addContentRenderView($renderView)
	{
		if (! empty($this->contentBuffer[ContentBuffer::CONTENT_TYPES_RENDER_VIEW])) {
			$this->contentBuffer[ContentBuffer::CONTENT_TYPES_RENDER_VIEW][0]->delete();
		}
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_RENDER_VIEW] = [];

		$contentBuffer = new ContentBuffer();
		$contentBuffer->setContent([
			"render" => $renderView
		]);
		$contentBuffer->type = ContentBuffer::CONTENT_TYPES_RENDER_VIEW;

		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_RENDER_VIEW][] = $contentBuffer;
	}

	public function cleanContentItem()
	{
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_ITEM] = [];
	}

	public function cleanContentChannel()
	{
		$this->contentBuffer[ContentBuffer::CONTENT_TYPES_CHANNEL] = [];
	}

	public function cleanContentSubscriberList()
	{
		$props = $this->diffusion->channel::CHANNEL_REQUIREMENTS['subscribers'];
		$props = array_keys($props);

		foreach ($this->contentBuffer[ContentBuffer::CONTENT_TYPES_SUBSCRIBER] as $contentBuffer) {
			array_walk($contentBuffer->content, function (&$subscriber) use ($props) {
				foreach ($subscriber->settings as $prop => $value) {
					if (! in_array($prop, $props)) {
						unset($subscriber->settings->{$prop});
					}
				}
			});
		}
	}

	public function previewView()
	{
		if (empty($this->contentHistory) || ! isset($this->contentHistory[ContentHistory::CONTENT_TYPES_RENDER_VIEW])) {
			return "";
		}

		$contentHistory = $this->contentHistory[ContentHistory::CONTENT_TYPES_RENDER_VIEW][0] ?? null;
		if (! is_string($contentHistory->content->render)) {
			return "";
		}
		return $contentHistory ? HTML::formatRender($contentHistory->content->render, $this->diffusion->name ?? "") : "";
	}

	private function updateDate()
	{
		$this->date = (new \DateTime())->format('Y-m-d H:i:s');
		$this->formatedDate = $this->getFormatedDate();
	}

	/**
	 * Permet creer/mettre a jour l'historique en fonction du statut passe
	 *
	 * @param integer $state
	 * @return mixed|void
	 * @throws \InvalidArgumentException
	 */
	public function state(int $state)
	{
		switch ($state) {
			case DiffusionHistory::INITIALIZED:
				if ($this->state != DiffusionHistory::INITIALIZED) {
					// L'historique precedent n'a pas ete envoye
					throw new \InvalidArgumentException("[DiffusionHistory] Invalid sequence state !");
				}
				$this->init();
				break;

			case DiffusionHistory::TO_VALIDATE:

				if ($this->state == DiffusionHistory::VALIDATED) {
					$this->state = DiffusionHistory::TO_VALIDATE;
					$this->update();
					break;
				}

				if ($this->state != DiffusionHistory::INITIALIZED) {
					throw new \InvalidArgumentException("[DiffusionHistory] Invalid sequence state !");
				}
				$this->toValidate();
				break;

			case DiffusionHistory::VALIDATED:
				if ($this->state != DiffusionHistory::TO_VALIDATE) {
					throw new \InvalidArgumentException("[DiffusionHistory] Invalid sequence state !");
				}
				$this->validate();
				break;

			case DiffusionHistory::SENT:
				if ($this->state != DiffusionHistory::VALIDATED) {
					throw new \InvalidArgumentException("[DiffusionHistory] Invalid sequence state !");
				}
				return $this->sent();

			case DiffusionHistory::RESET:
				if ($this->state != DiffusionHistory::TO_VALIDATE) {
					throw new \InvalidArgumentException("[DiffusionHistory] Invalid sequence state !");
				}
				$this->reset();
				break;

			default:
				throw new \InvalidArgumentException("[DiffusionHistory] Invalid state !");
		}
	}

	/**
	 * Permet de refaire l'envoi
	 *
	 * @throws \RuntimeException
	 * @return mixed
	 */
	public function send()
	{
		if (DiffusionHistory::SENT != $this->state) {
			throw new \RuntimeException("Invalid state change to state " . DiffusionHistory::VALIDATED);
		}

		$diffusionHistoryParser = new DiffusionHistoryParser();
		$contentHistoryRenderView = $this->contentHistory[ContentHistory::CONTENT_TYPES_RENDER_VIEW][0];
		$subscribers = $diffusionHistoryParser->parseSubscribers($this->contentHistory[ContentHistory::CONTENT_TYPES_SUBSCRIBER]);
		$channel = $diffusionHistoryParser->parseChannel($this->contentHistory[ContentHistory::CONTENT_TYPES_CHANNEL]);
		$channel->setTitle($this->diffusion->name);

		return $channel->send($subscribers, Helper::toArray($contentHistoryRenderView->content->render), $this->diffusion);
	}

	public function setDiffusion(Diffusion $diffusion)
	{
		$this->diffusion = $diffusion;
		$this->numDiffusion = $diffusion->id;
	}

	public function getPendingList()
	{
		$list = $this->getList([
			"state" => [
				"operator" => "not in",
				"value" => [DiffusionHistory::SENT, DiffusionHistory::NODATA],
			]
		], false, "state, date");

		for ($i = 0; $i < count($list); $i++) {

			// Optimisation on enleve les donnees des views que nous n'utilisons pas pour eviter les depassements memoires
			if (is_countable($list[$i]->contentBuffer) && count($list[$i]->contentBuffer) > 0) {
				unset($list[$i]->contentBuffer[ContentBuffer::CONTENT_TYPES_VIEW]);
			}

			if (! empty($list[$i]->diffusion->settings->isPrivate)) {
				array_splice($list, $i, 1);
				$i--;
				continue;
			}
		}
		return $list;
	}

	public function getChannel()
	{
		$diffusionHistoryParser = new DiffusionHistoryParser();
		return $diffusionHistoryParser->parseChannel($this->contentHistory[ContentHistory::CONTENT_TYPES_CHANNEL]);
	}

	public static function getInstance(int $id = 0, ?Diffusion $diffusion = null)
	{
		static::$instances[static::class] = static::$instances[static::class] ?? [];

		if (isset(static::$instances[static::class][$id])) {
			$instance = static::$instances[static::class][$id];
		} else {
			/** @phpstan-ignore new.static */
			$instance = new static($id, $diffusion);
			static::$instances[static::class][$id] = $instance;
		}

		return $instance;
	}

	/**
	 * Retourne la liste des diffusions publiques
	 *
	 * @return array
	 */
	public function getFilteredList($params)
	{
		$list = $this->getList($params);
		for ($i = 0; $i < count($list); $i++) {

			// Optimisation on enleve les donnees des views que nous n'utilisons pas pour eviter les depassements memoires
			if (is_countable($list[$i]->contentHistory) && count($list[$i]->contentHistory) > 0) {
				unset($list[$i]->contentHistory[ContentHistory::CONTENT_TYPES_VIEW]);
			}

			if (! empty($list[$i]->diffusion->settings->isPrivate)) {
				array_splice($list, $i, 1);
				$i--;
				continue;
			}
		}
		return $list;
	}

	/**
	 * Preparation des vues issues du buffer
	 * @return void
	 */
	protected function setupViewsBuffer()
	{
		if (! is_array($this->contentBuffer) || ! array_key_exists(ContentBuffer::CONTENT_TYPES_VIEW, $this->contentBuffer)) {
			return;
		}

		$viewsBuffer = $this->contentBuffer[ContentBuffer::CONTENT_TYPES_VIEW];
		$views = array();
		if (empty($viewsBuffer)) {
			return;
		}

		//On se fait un petit tableau plus facile a manipuler
		$rootView = 0;
		foreach ($viewsBuffer as $viewBuffer) {
			if ($viewBuffer->content && $viewBuffer->content->id) {
				$views[$viewBuffer->content->id] = $viewBuffer->content;
				if($viewBuffer->content->numParent == 0) {
					$rootView = $viewBuffer->content->id;
				}
			}
		}

		//On rend son enfant au parent
		foreach ($views as $view) {
			if ($view->numParent && array_key_exists($view->numParent, $views)) {
				if($rootView && $view->id != $rootView) {
					$views[$rootView]->childs[] = $view;
				}
				if(empty($views[$view->numParent]->childs)) {
					$views[$view->numParent]->childs[] = $view;
					continue;
				}
				$i = array_search($view->id, array_column($views[$view->numParent]->childs, "id"));
				if($i !== false) {
					$view = $views[$view->numParent]->childs[$i];
				}
			}
		}
	}

	protected function setupItemsBuffer()
	{
		if (! is_array($this->contentBuffer) || ! array_key_exists(ContentBuffer::CONTENT_TYPES_ITEM, $this->contentBuffer)) {
			return;
		}

		$itemsBuffer = $this->contentBuffer[ContentBuffer::CONTENT_TYPES_ITEM];
		$items = array();
		if (empty($itemsBuffer)) {
			return;
		}

		//On se fait un petit tableau plus facile a manipuler
		$rootItem = 0;
		foreach ($itemsBuffer as $itemBuffer) {
			if ($itemBuffer->content && $itemBuffer->content->id) {
				$items[$itemBuffer->content->id] = $itemBuffer->content;
				if($itemBuffer->content->numParent == 0) {
					$rootItem = $itemBuffer->content->id;
				}
			}
		}

		//On rend son enfant au parent
		foreach ($items as $item) {
			if ($item->numParent && array_key_exists($item->numParent, $items)) {
				if($rootItem && $item->id != $rootItem) {
					$items[$rootItem]->childs[] = $item;
				}
				if(empty($items[$item->numParent]->childs)) {
					$items[$item->numParent]->childs[] = $item;
					continue;
				}
				$i = array_search($item->id, array_column($items[$item->numParent]->childs, "id"));
				if($i !== false) {
					$item = $items[$item->numParent]->childs[$i];
				}
			}
		}
	}

	protected function updateImportViewBlocks(&$parentBlocks, &$childBlocks, $childId)
	{
		foreach($parentBlocks as $parentBlock) {
			if(!empty($parentBlock->blocks)) {
				$this->updateImportViewBlocks($parentBlock->blocks, $childBlocks, $childId);
			}
			if($parentBlock->type == WYSIWYGView::BLOCK_TYPES["viewImportInput"] && $parentBlock->content->viewId == $childId) {
				$childBlocks = $parentBlock->blocks;
			}
		}
	}

	/**
	 * Vide le buffer mais laisse intact $this->contentBuffer
	 */
	protected function clearBuffer()
	{
		if (empty($this->contentBuffer)) {
			$this->fetchContentBuffer();
		}
		$contentBufferOrm = ContentBufferOrm::find("num_diffusion_history", $this->id);
		while (count($contentBufferOrm) > 0) {
			$buffer = array_shift($contentBufferOrm);
			$buffer->delete();
		}
	}

	/**
	 * Vide le buffer et vide $this->contentBuffer
	 */
	protected function cleanBuffer()
	{
		$this->clearBuffer();
		$this->contentBuffer = [];
	}

	/**
	 * Pagination des diffusions en attente
	 *
	 * @param array $filters
	 * @param array $sort
	 * @param int $page
	 * @param int $nbPerPage
	 * @return array
	 */
	public static function paginationPending(array $filters = [], array $sort = [], int $page = 0, int $nbPerPage = 25): array
	{
		$clause = [];
		$needsDiffusionJoin = false;
		$needsTagJoin = false;
		$needsProductJoin = false;
		$needsContentBufferJoin = false;
		$contentBufferConditions = [];

		// Exclure les diffusions envoyees et sans donnees
		$clause[] = 'dsi_diffusion_history.state NOT IN (' . DiffusionHistory::SENT . ', ' . DiffusionHistory::NODATA . ')';

		// Filtre par nom de diffusion
		if (!empty($filters['diffusionName'])) {
			$searchTerm = addslashes($filters['diffusionName']);
			$clause[] = 'dsi_diffusion.name LIKE "%' . $searchTerm . '%"';
			$needsDiffusionJoin = true;
		}

		// Filtre par etat
		if (isset($filters['state']) && $filters['state'] !== '' && $filters['state'] !== null) {
			if (is_array($filters['state'])) {
				$filters['state'] = array_map('intval', $filters['state']);
				$clause[] = 'dsi_diffusion_history.state IN (' . implode(',', $filters['state']) . ')';
			} else {
				$clause[] = 'dsi_diffusion_history.state = ' . intval($filters['state']);
			}
		}
		
		// Filtre par tags
		if (!empty($filters['tags'])) {
			$tags = is_array($filters['tags']) ? array_map('intval', $filters['tags']) : [intval($filters['tags'])];
			$clause[] = 'dsi_entities_tags.num_tag IN (' . implode(',', $tags) . ')';
			$needsDiffusionJoin = true;
			$needsTagJoin = true;
		}

		// Filtre par produits
		if (!empty($filters['products'])) {
			$products = is_array($filters['products']) ? array_map('intval', $filters['products']) : [intval($filters['products'])];
			$clause[] = 'dsi_diffusion_product.num_product IN (' . implode(',', $products) . ')';
			$needsDiffusionJoin = true;
			$needsProductJoin = true;
		}

		// Filtre par type d'entite (type d'item)
		if (!empty($filters['entities'])) {
			$entities = is_array($filters['entities']) ? array_map('intval', $filters['entities']) : [intval($filters['entities'])];
			$entityConditions = [];
			foreach ($entities as $entityType) {
				// Recherche "type":X ou "type": X dans le JSON serialise
				$entityConditions[] = 'dsi_content_buffer.content LIKE \'%"type":' . $entityType . '%\'';
				$entityConditions[] = 'dsi_content_buffer.content LIKE \'%"type": ' . $entityType . '%\'';
			}
			$contentBufferConditions[] = '(dsi_content_buffer.type = ' . ContentBuffer::CONTENT_TYPES_ITEM . ' AND (' . implode(' OR ', $entityConditions) . '))';
			$needsContentBufferJoin = true;
		}

		// Filtre par canal (type de channel)
		if (!empty($filters['channel'])) {
			$channels = is_array($filters['channel']) ? array_map('intval', $filters['channel']) : [intval($filters['channel'])];
			$channelConditions = [];
			foreach ($channels as $channelType) {
				// Recherche "type":X ou "type": X dans le JSON serialise
				$channelConditions[] = 'dsi_content_buffer.content LIKE \'%"type":' . $channelType . '%\'';
				$channelConditions[] = 'dsi_content_buffer.content LIKE \'%"type": ' . $channelType . '%\'';
			}
			$contentBufferConditions[] = '(dsi_content_buffer.type = ' . ContentBuffer::CONTENT_TYPES_CHANNEL . ' AND (' . implode(' OR ', $channelConditions) . '))';
			$needsContentBufferJoin = true;
		}

		// Filtre par date
		if (!empty($filters['date'])) {
			if (!empty($filters['date']['start'])) {
				$clause[] = 'dsi_diffusion_history.date >= "' . addslashes($filters['date']['start']) . '"';
			}
			if (!empty($filters['date']['end'])) {
				$clause[] = 'dsi_diffusion_history.date <= "' . addslashes($filters['date']['end']) . ' 23:59:59"';
			}
		}

		// Filtre par abonnes
		// Recherche dans les champs name et email (email est dans settings)
		if (!empty($filters['subscribers'])) {
			$searchTerm = addslashes(trim($filters['subscribers']));
			$subscriberConditions = [
				"dsi_content_buffer.content LIKE '%\"name\":\"%{$searchTerm}%'",
				"dsi_content_buffer.content LIKE '%\"name\": \"%{$searchTerm}%'",
				"dsi_content_buffer.content LIKE '%\"email\":\"%{$searchTerm}%'",
				"dsi_content_buffer.content LIKE '%\"email\": \"%{$searchTerm}%'"
			];
			$contentBufferConditions[] = '(dsi_content_buffer.type = ' . ContentBuffer::CONTENT_TYPES_SUBSCRIBER . ' AND (' . implode(' OR ', $subscriberConditions) . '))';
			$needsContentBufferJoin = true;
		}

		// Gestion du tri
		$validColumns = [
			'date' => 'dsi_diffusion_history.date',
			'state' => 'dsi_diffusion_history.state',
			'diffusionName' => 'dsi_diffusion.name',
			'totalRecipients' => 'dsi_diffusion_history.total_recipients'
		];

		$sortClause = ' ORDER BY dsi_diffusion_history.state ASC, dsi_diffusion_history.date ASC';
		if (!empty($sort['name']) && isset($validColumns[$sort['name']])) {
			$column = $validColumns[$sort['name']];
			$sortDirection = (isset($sort['direction']) && $sort['direction'] === 'desc') ? 'DESC' : 'ASC';
			$sortClause = ' ORDER BY ' . $column . ' ' . $sortDirection;

			// Si on trie par nom de diffusion, on a besoin du JOIN
			if ($sort['name'] === 'diffusionName') {
				$needsDiffusionJoin = true;
			}
		}

		// Construction de la requete SQL
		$query = "SELECT DISTINCT dsi_diffusion_history.id_diffusion_history FROM dsi_diffusion_history";

		// Ajout du JOIN avec dsi_diffusion
		if ($needsDiffusionJoin) {
			$query .= " LEFT JOIN dsi_diffusion ON dsi_diffusion_history.num_diffusion = dsi_diffusion.id_diffusion";
		}

		// Ajout du JOIN avec dsi_entities_tags
		if ($needsTagJoin) {
			$query .= " LEFT JOIN dsi_entities_tags ON dsi_diffusion.id_diffusion = dsi_entities_tags.num_entity AND dsi_entities_tags.type = 1";
		}

		// Ajout du JOIN avec dsi_diffusion_product
		if ($needsProductJoin) {
			$query .= " LEFT JOIN dsi_diffusion_product ON dsi_diffusion.id_diffusion = dsi_diffusion_product.num_diffusion";
		}

		// Ajout des conditions de content_buffer via EXISTS (pour entities, channel, subscribers)
		// Chaque condition est une sous-requete EXISTS pour eviter les problemes avec plusieurs types
		if ($needsContentBufferJoin && !empty($contentBufferConditions)) {
			foreach ($contentBufferConditions as $condition) {
				$clause[] = 'EXISTS (SELECT 1 FROM dsi_content_buffer WHERE dsi_content_buffer.num_diffusion_history = dsi_diffusion_history.id_diffusion_history AND ' . $condition . ')';
			}
		}

		// Construction finale de la clause WHERE
		if (!empty($clause)) {
			$query .= ' WHERE ' . implode(' AND ', $clause);
		}

		// Ajout du tri
		$query .= $sortClause;

		// Pagination
		$pagination = DiffusionHistoryOrm::paginationFromQuery($query, $page, $nbPerPage);

		// Conversion des instances en modeles DiffusionHistory
		$historyList = [];
		foreach ($pagination['instances'] as $historyOrm) {
			$history = new DiffusionHistory($historyOrm->id_diffusion_history);

			// Filtrer les diffusions privees
			if (!empty($history->diffusion->settings->isPrivate)) {
				continue;
			}

			// Optimisation: on enleve les donnees volumineuses non necessaires pour la liste
			if (is_countable($history->contentBuffer) && count($history->contentBuffer) > 0) {
				unset($history->contentBuffer[ContentBuffer::CONTENT_TYPES_VIEW]);
			}

			$historyList[] = $history;
		}

		$pagination['instances'] = Helper::toArray($historyList);
		return $pagination;
	}

	/**
	 * Pagination des diffusions envoyees (historique)
	 *
	 * @param array $filters
	 * @param array $sort
	 * @param int $page
	 * @param int $nbPerPage
	 * @return array
	 */
	public static function paginationHistory(array $filters = [], array $sort = [], int $page = 0, int $nbPerPage = 25): array
	{
		$clause = [];
		$needsDiffusionJoin = false;
		$needsTagJoin = false;
		$needsProductJoin = false;
		$needsContentHistoryJoin = false;
		$contentHistoryConditions = [];

		// Uniquement les diffusions envoyees
		$clause[] = 'dsi_diffusion_history.state = ' . DiffusionHistory::SENT;

		// Filtre par nom de diffusion
		if (!empty($filters['diffusionName'])) {
			$searchTerm = addslashes($filters['diffusionName']);
			$clause[] = 'dsi_diffusion.name LIKE "%' . $searchTerm . '%"';
			$needsDiffusionJoin = true;
		}

		// Filtre par tags
		if (!empty($filters['tags'])) {
			$tags = is_array($filters['tags']) ? array_map('intval', $filters['tags']) : [intval($filters['tags'])];
			$clause[] = 'dsi_entities_tags.num_tag IN (' . implode(',', $tags) . ')';
			$needsDiffusionJoin = true;
			$needsTagJoin = true;
		}

		// Filtre par produits
		if (!empty($filters['products'])) {
			$products = is_array($filters['products']) ? array_map('intval', $filters['products']) : [intval($filters['products'])];
			$clause[] = 'dsi_diffusion_product.num_product IN (' . implode(',', $products) . ')';
			$needsDiffusionJoin = true;
			$needsProductJoin = true;
		}

		// Filtre par canal (type de channel)
		if (!empty($filters['channel'])) {
			$channels = is_array($filters['channel']) ? array_map('intval', $filters['channel']) : [intval($filters['channel'])];
			$channelConditions = [];
			foreach ($channels as $channelType) {
				$channelConditions[] = 'dsi_content_history.content LIKE \'%"type":' . $channelType . '%\'';
				$channelConditions[] = 'dsi_content_history.content LIKE \'%"type": ' . $channelType . '%\'';
			}
			$contentHistoryConditions[] = '(dsi_content_history.type = ' . ContentHistory::CONTENT_TYPES_CHANNEL . ' AND (' . implode(' OR ', $channelConditions) . '))';
			$needsContentHistoryJoin = true;
		}

		// Filtre par date
		if (!empty($filters['date'])) {
			if (!empty($filters['date']['start'])) {
				$clause[] = 'dsi_diffusion_history.date >= "' . addslashes($filters['date']['start']) . '"';
			}
			if (!empty($filters['date']['end'])) {
				$clause[] = 'dsi_diffusion_history.date <= "' . addslashes($filters['date']['end']) . ' 23:59:59"';
			}
		}

		// Filtre par abonnes
		if (!empty($filters['subscribers'])) {
			$searchTerm = addslashes(trim($filters['subscribers']));
			$subscriberConditions = [
				"dsi_content_history.content LIKE '%\"name\":\"%{$searchTerm}%'",
				"dsi_content_history.content LIKE '%\"name\": \"%{$searchTerm}%'",
				"dsi_content_history.content LIKE '%\"email\":\"%{$searchTerm}%'",
				"dsi_content_history.content LIKE '%\"email\": \"%{$searchTerm}%'"
			];
			$contentHistoryConditions[] = '(dsi_content_history.type = ' . ContentHistory::CONTENT_TYPES_SUBSCRIBER . ' AND (' . implode(' OR ', $subscriberConditions) . '))';
			$needsContentHistoryJoin = true;
		}

		// Gestion du tri
		$validColumns = [
			'date' => 'dsi_diffusion_history.date',
			'diffusionName' => 'dsi_diffusion.name',
			'totalRecipients' => 'dsi_diffusion_history.total_recipients'
		];

		$sortClause = ' ORDER BY dsi_diffusion_history.date DESC';
		if (!empty($sort['name']) && isset($validColumns[$sort['name']])) {
			$column = $validColumns[$sort['name']];
			$sortDirection = (isset($sort['direction']) && $sort['direction'] === 'desc') ? 'DESC' : 'ASC';
			$sortClause = ' ORDER BY ' . $column . ' ' . $sortDirection;

			if ($sort['name'] === 'diffusionName') {
				$needsDiffusionJoin = true;
			}
		}

		// Construction de la requete SQL
		$query = "SELECT DISTINCT dsi_diffusion_history.id_diffusion_history FROM dsi_diffusion_history";

		if ($needsDiffusionJoin) {
			$query .= " LEFT JOIN dsi_diffusion ON dsi_diffusion_history.num_diffusion = dsi_diffusion.id_diffusion";
		}

		if ($needsTagJoin) {
			$query .= " LEFT JOIN dsi_entities_tags ON dsi_diffusion.id_diffusion = dsi_entities_tags.num_entity AND dsi_entities_tags.type = 1";
		}

		if ($needsProductJoin) {
			$query .= " LEFT JOIN dsi_diffusion_product ON dsi_diffusion.id_diffusion = dsi_diffusion_product.num_diffusion";
		}

		if ($needsContentHistoryJoin && !empty($contentHistoryConditions)) {
			foreach ($contentHistoryConditions as $condition) {
				$clause[] = 'EXISTS (SELECT 1 FROM dsi_content_history WHERE dsi_content_history.num_diffusion_history = dsi_diffusion_history.id_diffusion_history AND ' . $condition . ')';
			}
		}

		if (!empty($clause)) {
			$query .= ' WHERE ' . implode(' AND ', $clause);
		}

		$query .= $sortClause;

		$pagination = DiffusionHistoryOrm::paginationFromQuery($query, $page, $nbPerPage);

		$historyList = [];
		foreach ($pagination['instances'] as $historyOrm) {
			$history = new DiffusionHistory($historyOrm->id_diffusion_history);

			if (!empty($history->diffusion->settings->isPrivate)) {
				continue;
			}

			if (is_countable($history->contentHistory) && count($history->contentHistory) > 0) {
				unset($history->contentHistory[ContentHistory::CONTENT_TYPES_VIEW]);
			}

			$historyList[] = $history;
		}

		$pagination['instances'] = Helper::toArray($historyList);
		return $pagination;
	}
}
