<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: AzureOAuth.php,v 1.3.4.1 2026/02/04 11:11:17 dbellamy Exp $

namespace Pmb\Common\Library\Mailer;

use League\OAuth2\Client\Token\AccessToken;
use PHPMailer\PHPMailer\OAuth;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use \Pmb\Common\Helper\DateHelper;
use mail;

class AzureOAuth extends OAuth
{

    /**
     * Duree de validite refresh token Azure (en jours)
     */
    const REFRESH_TOKEN_DURATION = 90;

    /**
     * Duree de validite minimum refresh token Azure (en jours)
     */
    const MIN_REFRESH_TOKEN_DURATION = 30;

    protected $options = [];

    protected $context = '';


    public function __construct($provider, $options)
    {
        $this->provider = $provider;

        $this->oauthUserEmail = $options['user'];
        $this->oauthClientSecret = $options['xoauth2_secret_value'];
        $this->oauthClientId = $options['xoauth2_client_id'];
        $this->oauthRefreshToken = $options['xoauth2_refresh_token'];

        $this->options = $options;
        $this->options['xoauth2_refresh_token_validity'] = $options['xoauth2_refresh_token_validity'] ?? '';
        $this->context = (defined('GESTION')) ? 'pmb' : 'opac';
        $this->options['id'] = $options['id'] ?? $this->context;
    }

    /**
     * Mise a jour du refresh token
     *
     * @return void
     */
    protected function updateRefreshToken()
    {
        $dt_now = new \DateTime('now');

        try {
            $xoauth2_refresh_token_validity = pmb_preg_replace("[^0-9]", '', $this->options['xoauth2_refresh_token_validity']);
            $dt_token = new \DateTime($xoauth2_refresh_token_validity);
            $diff = ceil(DateHelper::getDiffInSeconds($dt_now, $dt_token) / 86400);
            if($diff > AzureOAuth::MIN_REFRESH_TOKEN_DURATION ) {
                return;
            }
        } catch (\Exception $e) {
        }

        $this->options['xoauth2_old_refresh_token'] = $this->options['xoauth2_refresh_token'];

        $new_dt_token = clone $dt_now;
        $new_dt_token->add(new \DateInterval('P'.AzureOAuth::REFRESH_TOKEN_DURATION.'D'));

        $this->options['xoauth2_refresh_token'] = $this->oauthToken->getRefreshToken();
        $this->options['xoauth2_refresh_token_validity'] = $new_dt_token->format('Y-m-d');

        mail::updateRefreshTokens($this->options);
    }

    /**
     * fetch a new token if it's not available or has expired
     *
     * @return AccessToken
     */
    protected function fetchOauthToken()
    {
        if (null === $this->oauthToken || $this->oauthToken->hasExpired()) {
            $this->oauthToken = $this->getToken();
        }

        $this->updateRefreshToken();
        return $this->oauthToken;
    }

    /**
     * Generate a base64-encoded OAuth token.
     *
     * @return string
     */
    public function getOauth64()
    {
        return base64_encode(
            'user=' .
            $this->oauthUserEmail .
            "\001auth=Bearer " .
            $this->fetchOauthToken() .
            "\001\001"
            );
    }
}

