<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: SEOSitemapEntry.php,v 1.10.2.2 2026/01/09 15:16:01 rtigero Exp $

namespace Pmb\SEO\Models;

use encoding_normalize;
use Pmb\Common\Helper\Helper;
use Pmb\Common\Models\Model;

abstract class SEOSitemapEntry extends Model
{
    protected $ormName = "Pmb\SEO\Orm\SEOSitemapEntryOrm";
    public $idSitemapEntry = 0;
    public $type = "";
    public $inSitemap = true;
    public $priority = 0.00;
    public $frequencyUpdate = "";
    public $sitemapOrder = 0;
    public $settings = "";
    public $numParent = 0;

    public $childs = array();

    public function __construct(int $id = 0)
    {
        parent::__construct($id);
        if (empty($this->settings)) {
            $this->initDefaultSettings();
        } elseif (is_string($this->settings)) {
            $this->settings = encoding_normalize::json_decode($this->settings, true);
        }
    }

    /**
     * Initialise les paramtres par dfaut pour l'entre de sitemap
     * Doit tre implment par les classes enfants
     */
    protected function initDefaultSettings(): void
    {
        // Mthode  implmenter par les classes enfants
        $this->settings = array();
    }

    /**
     * Liste tous les types de sitemap en retournant les entits concernes
     *
     * @return array
     */
    final public static function getSitemapEntries(): array
    {
        $ret = array();
        $files = glob(__DIR__ . DIRECTORY_SEPARATOR . 'Entries' . DIRECTORY_SEPARATOR . '*.php');
        foreach ($files as $file) {
            $class = str_replace('.php', '', basename($file));
            $fullClass = "Pmb\\SEO\\Models\\Entries\\" . $class;
            if (class_exists($fullClass) && is_subclass_of($fullClass, self::class)) {
                $defaultEntry = new $fullClass();
                $list = array();
                $ormList = $defaultEntry->ormName::find("type", $defaultEntry->type, "sitemap_order");
                foreach ($ormList as $ormEntry) {
                    $list[] = new $fullClass($ormEntry->{$defaultEntry->ormName::$idTableName});
                }
                $ret[] = array(
                    "default_entry" => $defaultEntry,
                    "list" => $list
                );
            }
        }
        return $ret;
    }

    /**
     * Remplit $this->childs avec les enfants de l'lment
     * Peut initaliser en base les children selon le type d'entre
     */
    public function initChildren(): void
    {
        $this->fetchChildren();
    }

    /**
     * Remplit $this->childs avec les enfants de l'lment
     * Rcupre les childs en base mais ne les initalise pas s'ils ne sont pas en base
     */
    final public function fetchChildren($recursive = false): void
    {
        if ($this->id && empty($this->childs)) {
            $childs = $this->ormName::find("num_parent", $this->id, "sitemap_order");
            foreach ($childs as $child) {
                $child = new static($child->{$this->ormName::$idTableName});
                if ($recursive) {
                    $child->fetchChildren($recursive);
                }
                $this->childs[] = $child;
            }
        }
    }

    abstract public function check($data): array;

    public function setFromForm($data): void
    {
        // Mthode  implmenter par les classes enfants
        $this->priority = floatval($data->priority);
        $this->frequencyUpdate = $data->frequencyUpdate;
        $this->sitemapOrder = intval($data->sitemapOrder);
        $this->inSitemap = boolval($data->inSitemap);
        $this->settings = Helper::toArray($data->settings) ?? array();
        $this->numParent = intval($data->numParent);
    }

    public function create(): void
    {
        $orm = new $this->ormName();
        $orm->type = $this->type;
        $orm->in_sitemap = $this->inSitemap;
        $orm->priority = $this->priority;
        $orm->frequency_update = $this->frequencyUpdate;
        $orm->sitemap_order = $this->sitemapOrder;
        $orm->settings = encoding_normalize::json_encode($this->settings);
        $orm->num_parent = $this->numParent;
        $orm->save();

        $this->id = $orm->{$this->ormName::$idTableName};
        $this->{Helper::camelize($this->ormName::$idTableName)} = $orm->{$this->ormName::$idTableName};
    }

    public function update(): void
    {
        $orm = new $this->ormName($this->id);
        $orm->type = $this->type;
        $orm->in_sitemap = $this->inSitemap;
        $orm->priority = $this->priority;
        $orm->frequency_update = $this->frequencyUpdate;
        $orm->sitemap_order = $this->sitemapOrder;
        $orm->num_parent = $this->numParent;
        $orm->settings = encoding_normalize::json_encode($this->settings);
        $orm->save();
    }

    /**
     * Supprime l'entre et ses enfants
     *
     * @param bool $updateOrder Si true, met  jour l'ordre des frres et soeurs
     */
    public function delete($updateOrder = true)
    {
        //On supprime les enfants de manire rcursive en passant par le modle
        $childs = $this->ormName::find("num_parent", $this->id);
        foreach ($childs as $child) {
            $childModel = new static($child->{$this->ormName::$idTableName});
            //On supprime tous les enfants donc inutile de s'embeter avec l'ordre
            $childModel->delete(false);
        }


        $orm = new $this->ormName($this->id);

        $orm->delete();

        if ($updateOrder) {
            $this->updateOrder();
        }

        $this->id = 0;
        $this->{Helper::camelize($orm::$idTableName)} = 0;
        $this->inSitemap = true;
        $this->priority = 0.00;
        $this->frequencyUpdate = "";
        $this->sitemapOrder = 0;
        $this->settings = "";
        $this->numParent = 0;

        return [
            'error' => false,
            'errorMessage' => '',
        ];
    }

    /**
     * Suppression rcursive des enfants de l'lment
     */
    public function clearChildren(): void
    {
        $this->initChildren();
        foreach ($this->childs as $child) {
            $child->delete(false);
        }
        $this->childs = array();
    }

    /**
     * Mise  jour de l'ordre des frres et soeurs du mme niveau en cas de suppression de l'lment
     */
    protected function updateOrder(): void
    {
        //On rcupre tous les frres et soeurs pour rordonner
        //Venant aprs dans l'ordre
        $siblings = $this->ormName::finds(array(
            "num_parent" => $this->numParent,
            $this->ormName::$idTableName => array(
                "value" => $this->id,
                "operator" => "!="
            ),
            "sitemap_order" => array(
                "value" => $this->sitemapOrder,
                "operator" => ">"
            )
        ), "sitemap_order");

        foreach ($siblings as $sibling) {
            $siblingOrm = new $this->ormName($sibling->{$this->ormName::$idTableName});
            $siblingOrm->sitemap_order = $siblingOrm->sitemap_order - 1;
            $siblingOrm->save();
        }
    }

    /**
     * Suppression et recalcul rcursif des enfants
     */
    public function resetChilds(): void
    {
        $this->clearChildren();
        $this->initChildren();

        foreach ($this->childs as $child) {
            $child->resetChilds();
        }
    }

    /**
     * Rinitialise la hirarchie complte du sitemap en partant des parents
     */
    public static function resetHierarchy(): void
    {
        $model = new static();
        $topHierarchy = $model->ormName::find("num_parent", 0, "sitemap_order");
        //Le reset tant rcursif, il suffit de passer sur les parents pour tout recalculer
        foreach ($topHierarchy as $entry) {
            $modelNamespace = "Pmb\\SEO\\Models\\Entries\\SEOSitemapEntry" . ucfirst($entry->type);
            if (class_exists($modelNamespace)) {
                $entryObj = new $modelNamespace($entry->{$entry::$idTableName});
                $entryObj->resetEntryHierarchy();
            }
        }
    }

    /**
     * Retourne l'URL de l'entry pour le sitemap
     * A driver selon le type d'entre
     */
    abstract public function getEntryUrl(): string;

    /**
     * Recalcule la hirarchie des enfants de l'entre
     */
    abstract public function resetEntryHierarchy(): void;

    /**
     * Retourne la date de dernire modification de l'entre
     *
     * @return string Date de dernire modification
     */
    abstract public function getLastMod(): string;
}
