<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: OcrExplnumsModel.php,v 1.1.2.1 2025/10/21 08:22:46 qvarin Exp $

namespace Pmb\Ocr\Models;

if (stristr($_SERVER['REQUEST_URI'], basename(__FILE__))) {
    die("no access");
}

use Curl;
use explnum;
use Pmb\Common\Models\Model;
use Pmb\Docnums\Models\Explnum as ExplnumCollection;
use Pmb\Ocr\Library\Ocr\OCRMyPdf;
use Pmb\Ocr\Library\Server\OcrServer;
use Pmb\Ocr\Orm\OcrExplnumsOrm;

class OcrExplnumsModel extends Model
{
    public const TYPE_EXPLNUM = 1;
    public const TYPE_EXPLNUM_COLLECTION = 2;

    /**
     * Correspond au namespace de l'ORM
     *
     * @var string
     */
    protected $ormName = "Pmb\\Ocr\\Orm\\OcrExplnumsOrm";

    public $idOcrExplnums = 0;
    public $explnumNum = 0;
    public $explnumCollectionsExplnumsNum = 0;
    public $ocrExplnumsHash = "";

    /**
     * Vrifie si le document est trait
     *
     * @return int
     */
    protected function getState(): int
    {
        $ocrServer = new OcrServer();
        if (!$ocrServer->isActive()) {
            return OcrServer::STATE_UNKNOWN;
        }

        $result = $ocrServer->state($this->ocrExplnumsHash);
        if ($result['success'] !== true) {
            return OcrServer::STATE_UNKNOWN;
        }
        return $result['data']['state'] ?? OcrServer::STATE_UNKNOWN;
    }

    /**
     * Rcupre le document ocr
     *
     * @return string|false
     */
    protected function pullDocument()
    {
        $ocrServer = new OcrServer();
        if (!$ocrServer->isActive()) {
            return false;
        }

        $result = $ocrServer->pull($this->ocrExplnumsHash);
        if ($result['success'] === true) {
            return $result['data'];
        }
        return false;
    }

    /**
     * Met  jour le document
     *
     * @param string $binary
     * @return boolean
     */
    public function update(string $binary): bool
    {
        if ($this->explnumNum > 0) {
            // On met  jour un explnum
            $explnum = new explnum($this->explnumNum);
        } else {
            // On met  jour un explnum d'une collection
            $explnum = new ExplnumCollection($this->explnumCollectionsExplnumsNum);
        }

        return $explnum->updateBinary($binary);
    }

    /**
     * Supprime l'instance
     *
     * @return void
     */
    protected function delete(): void
    {
        $orm = new OcrExplnumsOrm($this->idOcrExplnums);
        $orm->delete();
    }

    public static function isOcrByServer(): bool
    {
        $ocrServer = new OcrServer();
        return $ocrServer->isActive();
    }

    public static function isOcrByOcrMyPdf(): bool
    {
        $ocrMyPdf = new OCRMyPdf();
        return $ocrMyPdf->isActive();
    }

    /**
     * Get the instance
     *
     * @param int $type
     * @param int $explnumId
     * @param string $filename
     * @return bool
     */
    public static function exist(int $type, int $explnumId, string $filename): bool
    {
        $fields = [
            "explnum_num" => 0,
            "explnum_collections_explnums_num" => 0,
            "ocr_explnums_hash" => md5_file($filename),
        ];

        if ($type === self::TYPE_EXPLNUM) {
            $fields["explnum_num"] = $explnumId;
        } else {
            $fields["explnum_collections_explnums_num"] = $explnumId;
        }

        $instances = OcrExplnumsOrm::finds($fields);
        if (!empty($instances)) {
            return true;
        }
        return false;
    }

    /**
     * Ajoute un document  traiter
     *
     * @param int $type
     * @param int $explnumId
     * @param string $filename
     * @return void
     */
    public static function push(int $type, int $explnumId, string $filename)
    {
        $ocrServer = new OcrServer();
        if (!$ocrServer->isActive() || !is_file($filename)) {
            return;
        }

        if (self::exist($type, $explnumId, $filename)) {
            return;
        }

        $result = $ocrServer->push($filename);
        if ($result['success'] === true) {
            $orm = new OcrExplnumsOrm();
            if ($type === self::TYPE_EXPLNUM) {
                $orm->explnum_num = $explnumId;
                $orm->explnum_collections_explnums_num = 0;
            } else {
                $orm->explnum_num = 0;
                $orm->explnum_collections_explnums_num = $explnumId;
            }
            $orm->ocr_explnums_hash = $result['data']['hash'];
            $orm->save();
        }
    }

    /**
     * Rcupre le document  traiter
     *
     * @return self|null
     */
    public static function next(): ?self
    {
        $instances = OcrExplnumsOrm::finds([], "id_ocr_explnums ASC", "1");
        if (empty($instances)) {
            return null;
        }

        return new self($instances[0]->id_ocr_explnums);
    }

    /**
     * Lance l'ocrisation
     *
     * @return void
     */
    protected static function initOcrExplnums()
    {
        global $pmb_url_internal;

        $curl = new Curl();
		$curl->set_option('CURLOPT_TIMEOUT', '1');
		$curl->set_option('CURLOPT_SSL_VERIFYPEER', false);
		$curl->get($pmb_url_internal . 'ocr_stack.php?database=' . LOCATION);
    }

    /**
     * Indique le nombre de documents  traiter
     *
     * @return string
     */
    public static function getOcrState(): string
    {
        if (!self::isOcrByServer()) {
            return '';
        }

        $result = pmb_mysql_query("SELECT count(*) FROM ocr_explnums");
        if (pmb_mysql_num_rows($result) > 0) {
            $count = pmb_mysql_result($result, 0);
            pmb_mysql_free_result($result);

            if ($count > 0) {
                self::initOcrExplnums();

                global $msg;
                return sprintf($msg['ocr_state'], $count);
            }
        }

        return '';
    }

    /**
     * Traite les documents
     *
     * @return void
     */
    public static function process()
    {
        $model = self::next();
        if (empty($model)) {
            // On a fini
            return;
        }

        $state = $model->getState();
        if (in_array($state, [OcrServer::STATE_PENDING, OcrServer::STATE_IN_PROGRESS])) {
            // Le document n'est pas trait ou en cours de traitement
            return;
        }

        if ($state === OcrServer::STATE_COMPLETED) {
            $binary = $model->pullDocument();
            if (false !== $binary) {
                $model->update($binary);
            }
        }

        $model->delete();
        self::process();
    }
}
