<?php
// +-------------------------------------------------+
// | 2002-2007 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: RdfConverter.php,v 1.9.2.2 2025/06/27 14:21:08 rtigero Exp $
namespace Pmb\ImportExport\Library;

use rdf_entities_converter_controller;

class RdfConverter
{

    protected array $entities = [];

    protected array $convertedEntities = [];

    /**
     * Tableau permettant  la classe de connaitre toutes les entits  importer
     * en cas d'utilisation sous forme de chunk
     */
    protected array $rootEntities = [];

    protected string $baseUrl = "http://opac.local/";

    protected int $depth = 5;

    public const MIN_DEPTH = 0;
    public const MAX_DEPTH = 5;

    protected $store = null;

    public array $rdfAssertions = [];

    public string $rdf = '';

    protected array $ns = [
        "dc" => "http://purl.org/dc/elements/1.1",
        "dct" => "http://purl.org/dc/terms/",
        "owl" => "http://www.w3.org/2002/07/owl#",
        "rdf" => "http://www.w3.org/1999/02/22-rdf-syntax-ns#",
        "rdfs" => "http://www.w3.org/2000/01/rdf-schema#",
        "skos" => "http://www.w3.org/2004/02/skos/core#",
        "xsd" => "http://www.w3.org/2001/XMLSchema#",
        "pmb" => "http://www.pmbservices.fr/ontology#"
    ];

    protected array $depthByEntities = [];


    /**
     *
     * @param array $entities : ['type'=> 'entity_type', 'id' => 'entity_id] : tableau d'entites type+id
     * @param string $baseUrl : URL de base pour construction des assertions
     * @param int $depth : profondeur d'exploration
     * @param $store :
     *            entrepot Rdf
     */
    public function __construct(array $entities = [], string $baseUrl = '', int $depth = 5, $store = null, $depthByEntities = array())
    {
        $this->entities = $entities;
        $this->baseUrl = $baseUrl;
        $this->depth = $depth;
        $this->store = $store;
        $this->depthByEntities = $depthByEntities;
    }

    /**
     * Ajoute des entites a la liste a parcourir
     *
     * @param array $entities
     */
    public function addEntities(array $entities = [])
    {
        $this->entities = array_merge($this->entities, $entities);
    }

    /**
     * Vide les donnes de l'instance
     *
     */
    public function clear()
    {
        $this->entities = array();
        $this->rdfAssertions = array();
        $this->rdf = "";
    }

    /**
     * Parcours des entites
     */
    public function run($generate_mode = false)
    {

        foreach ($this->entities as $id => $entity) {
            $this->entities[$id]['depth'] = 0;
            if (!empty($this->depthByEntities)) {
                $this->entities[$id]['depthToGo'] = $this->depthByEntities[$entity['type']];
            }
        }

        while (! empty($this->entities)) {

            $entity = array_shift($this->entities);
            // var_dump($entity);
            $entityType = $entity['type'];
            $entityId = $entity['id'];
            $entityDepth = $entity['depth'];
            if (!empty($this->depthByEntities)) {
                $depthToGo = $entity['depthToGo'] ?? $this->depthByEntities[$entityType];
                if ($depthToGo === false) {
                    continue;
                }
            }

            //On force la profondeur  0 pour les entits racine
            if (array_key_exists($entityType, $this->rootEntities) && in_array($entityId, $this->rootEntities[$entityType])) {
                $entityDepth = 0;
            }

            if (isset($this->convertedEntities[$entityType]) && in_array($entityId, $this->convertedEntities[$entityType])) {
                continue;
            }
            $this->convertedEntities[$entityType][] = $entityId;

            $converterName = rdf_entities_converter_controller::get_entity_converter_name_from_type($entityType);
            if (is_null($converterName)) {
                continue;
            }
            $converter = new $converterName($entityId, $entityType, $this->baseUrl . $entityType . '#' . $entityId, 1, $this->store);

            $assertions = $converter->get_assertions();
            // var_dump($assertions);

            $this->browseAssertions($entityType, $entityDepth, $assertions, $depthToGo ?? 0);
            $assertions = null;
            $converter = null;
        }
        if ($generate_mode) {
            return $this->generateRdfHeader() . $this->rdf . $this->generateRdfFooter();
        }
    }


    protected function browseAssertions(string $entityType = 'record', int $depth = 0, array $assertions = [], $depthToGo = 0)
    {
        if ($depth > $this->depth) {
            return;
        }
        while (! empty($assertions)) {

            $assertion = array_shift($assertions);

            $subject = $assertion->get_subject();

            if (! array_key_exists($subject, $this->rdfAssertions)) {
                $this->rdfAssertions[$subject] = [
                    'about' => $subject,
                    'type' => 'pmb:' . $entityType,
                    'properties' => [
                        "pmb:depth" => array($depth)
                    ],
                    'resources' => []
                ];
            }

            $predicate = $assertion->get_predicate();
            $predicate = str_replace("http://www.pmbservices.fr/ontology#", "pmb:", $predicate);

            $objectType = strtolower($assertion->get_object_type());
            $explodedObjectType = explode("#", $objectType);
            $pmbObjectType = "";
            if (count($explodedObjectType) > 1 && array_key_exists($explodedObjectType[1], $this->depthByEntities)) {
                $pmbObjectType = $explodedObjectType[1];
            }
            $object = $assertion->get_object();
            $fake_object = $object;
            $objectProperties = $assertion->get_object_properties();

            // Petit hack pour les concepts car l'ID et l'URI ne correspondent pas
            if (!empty($this->depthByEntities) && $this->depthByEntities["concept"] !== false) {
                if (($predicate == "pmb:has_concept") && ! empty($object)) {
                    $concept_uri = \onto_common_uri::get_uri($object);
                    $pos = strrpos($concept_uri, '#');
                    if (false !== $pos) {
                        $fake_object = substr($concept_uri, $pos + 1);
                    }
                }
            }
            // $assertion = null;

            switch (true) {

                // Objet non dfini
                case (($object == '') || is_null($object)):
                    break;

                // Objet de type literal
                case ($objectProperties['type'] == 'literal'):

                    $this->addLiteral($subject, $predicate, $object);
                    break;

                // Objet de type URI sans noeud blanc
                case (($objectProperties['type'] == 'uri') && empty($objectProperties['object_assertions'])):
                    //On ne prend pas les liens si on est au bout de la profondeur ou si l'objet ne fait pas partie des
                    //Entits  exporter
                    if ((!empty($this->depthByEntities) && $depthToGo == 0) || ($pmbObjectType != "" && $this->depthByEntities[$pmbObjectType] === false)) {
                        break;
                    }

                    // Lien entre l'entite et la ressource
                    if (isset($objectProperties["identifier"])) {
                        //Cas des autorits lies notamment
                        $fake_object = $objectProperties["identifier"];
                        $object = $objectProperties["identifier"];
                    }
                    if (stripos($objectType, "http://www.pmbservices.fr/ontology#") !== false) {
                        $resourceSubject = $this->baseUrl . str_replace("http://www.pmbservices.fr/ontology#", "", $objectType) . "#" . $object;
                    } else if (stripos($objectType, "http://www.w3.org/2004/02/skos/core#") !== false) {
                        $resourceSubject = $this->baseUrl . str_replace("http://www.w3.org/2004/02/skos/core#", "", "skos/" . $objectType) . "#" . $fake_object;
                    }
                    $this->addResourceLink($subject, $predicate, $resourceSubject);

                    // Creation ressource
                    $resourceId = $object;
                    $resourceType = $objectType;
                    $resourceProperties = $objectProperties;
                    $this->addResource($resourceId, $resourceType, $resourceSubject, $resourceProperties, $depth, $depthToGo - 1);
                    break;

                // Objet de type URI avec noeud blanc
                case (($objectProperties['type'] == 'uri') && ! empty($objectProperties['object_assertions']) && ! empty($objectProperties['sub_uri'])):
                    //On ne prend pas les liens si on est au bout de la profondeur ou si l'objet ne fait pas partie des
                    //Entits  exporter
                    if ((!empty($this->depthByEntities) && $depthToGo == 0) || ($pmbObjectType != "" && $this->depthByEntities[$pmbObjectType] === false)) {
                        break;
                    }
                    // Lien entre l'entite et le noeud blanc
                    if (stripos($objectType, "http://www.pmbservices.fr/ontology#") !== false) {
                        $bnodeSubject = $this->baseUrl . str_replace("http://www.pmbservices.fr/ontology#", "", $objectType) . "#" . $object;
                    } else if (stripos($objectType, "http://www.w3.org/2004/02/skos/core#") !== false) {
                        $bnodeSubject = $this->baseUrl . str_replace("http://www.w3.org/2004/02/skos/core#", "", "skos/" . $objectType) . "#" . $fake_object;
                    }
                    $this->addResourceLink($subject, $predicate, $bnodeSubject);

                    // Creation noeud blanc
                    // Noeud blanc
                    $bnodeId = $object;
                    $bnodeType = $objectType;
                    $bnodeProperties = $objectProperties;
                    $this->addBnode($bnodeId, $bnodeType, $bnodeSubject, $bnodeProperties, $depth);
                    break;

                default:
                    break;
            }
            $subject = null;
            $predicate = null;
            $objectProperties = null;
        }
    }


    /**
     * Ajoute un literal aux assertions RDF
     *
     * @param string $subject
     * @param string $predicate
     * @param string $literal
     */
    protected function addLiteral(string $subject, string $predicate, string $literal)
    {
        $this->rdfAssertions[$subject]['properties'][$predicate][] = $literal;
    }

    /**
     * Ajoute une ressource
     *
     * @param string $resourceId
     * @param string $objectType
     * @param array $objectProperties
     * @param string $resourceSubject
     */
    protected function addResource(string $resourceId, string $resourceType, string $resourceSubject, array $resourceProperties, int $depth, int $depthToGo)
    {
        if (array_key_exists($resourceSubject, $this->rdfAssertions)) {
            return;
        }
        $resType = str_replace("http://www.pmbservices.fr/ontology#", "", $resourceType);
        $resType = str_replace("http://www.w3.org/2004/02/skos/core#", "", $resType);
        $resType = strtolower($resType);
        $converter = rdf_entities_converter_controller::get_entity_converter_name_from_type($resType);
        // Pas d'exploration en profondeur ou pas de converter associe
        // on ajoute directement l'entite dans rdfAssertions
        if ((0 == $this->depth) || is_null($converter)) {
            $resObject = isset($resourceProperties['display_label']) ? $resourceProperties['display_label'] : $resourceId;
            $this->rdfAssertions[$resourceSubject] = [
                'about' => $resourceSubject,
                'type' => "pmb:" . $resType,
                'properties' => [],
                'resources' => []
            ];
            $this->rdfAssertions[$resourceSubject]['properties']['pmb:displayLabel'][] = $resObject;
        } else {
            // Si la ressource est une entite et a un converter,
            // on l'ajoute a la liste des entites a parcourir
            $newEntity = [
                'type' => $resType,
                'id' => $resourceId,
                'depth' => $depth + 1,
            ];
            if (!empty($this->depthByEntities)) {
                $newEntity['depthToGo'] = max($this->depthByEntities[$resType], $depthToGo);
            }
            $this->entities[] = $newEntity;
        }
    }

    /**
     * Ajoute un lien entre une entite et 1 ressource
     *
     * @param string $subject
     * @param string $predicate
     * @param string $object
     */
    protected function addResourceLink(string $subject, string $predicate, string $object)
    {
        $this->rdfAssertions[$subject]['resources'][$predicate][] = $object;
    }

    /**
     * Ajoute un noeud blanc
     */
    protected function addBnode(string $bnodeId, string $bnodeType, string $bnodeSubject, array $bnodeProperties, $depth)
    {
        if (array_key_exists($bnodeSubject, $this->rdfAssertions)) {
            return;
        }

        $bnType = str_replace("http://www.pmbservices.fr/ontology#", "", $bnodeType);
        $bnType = str_replace("http://www.w3.org/2004/02/skos/core#", "", $bnType);
        $bnType = strtolower($bnType);
        $converter = rdf_entities_converter_controller::get_entity_converter_name_from_type($bnType);

        // Pas d'exploration en profondeur ou pas de converter associe
        // on ajoute directement l'entite dans rdfAssertions
        if ((0 == $this->depth) || is_null($converter)) {

            $this->rdfAssertions[$bnodeSubject] = [
                'about' => $bnodeSubject,
                'type' => '',
                'properties' => [],
                'resources' => []
            ];

            if (is_countable($bnodeProperties['object_assertions'])) {

                $bnAssertions = $bnodeProperties['object_assertions'];

                foreach ($bnAssertions as $bnAssertion) {

                    $predicate = $bnAssertion->get_predicate();
                    $predicate = str_replace("http://www.pmbservices.fr/ontology#", "pmb:", $predicate);

                    $objectType = $bnAssertion->get_object_type();
                    $object = $bnAssertion->get_object();
                    $objectProperties = $bnAssertion->get_object_properties();

                    switch (true) {

                        // Objet de type literal
                        case (($objectProperties['type'] == 'literal') && ($object !== '')):

                            $this->addLiteral($bnodeSubject, $predicate, $object);
                            break;

                        // Objet de type URI sans noeud blanc
                        case (($objectProperties['type'] == 'uri') && ($object !== '') && empty($objectProperties['object_assertions'])):

                            // Lien entre l'entite et la ressource
                            $resourceSubject = $this->baseUrl . str_replace("http://www.pmbservices.fr/ontology#", "", $objectType) . "#" . $object;

                            $this->addResourceLink($bnodeSubject, $predicate, $resourceSubject);

                            // Creation ressource
                            $resourceId = $object;
                            $resourceType = $objectType;
                            $resourceProperties = $objectProperties;
                            $this->addResource($resourceId, $resourceType, $resourceSubject, $resourceProperties, $depth, 1);
                            break;

                        default:
                            break;
                    }
                }
            }
        } else {

            // Si la ressource est une entite et a un converter,
            // on l'ajoute a la liste des entites a parcourir
            $newEntity = [
                'type' => $bnType,
                'id' => $bnodeId,
                'depth' => $depth,
            ];
            if (!empty($this->depthByEntities)) {
                //On met 1 car le noeud blanc ne contient qu'un niveau
                $newEntity['depthToGo'] = 1;
            }
            $this->entities[] = $newEntity;
        }
    }

    /**
     * Genere un RDF/XML a partir des assertions RDF
     *
     * @param string $format : full|compact
     *
     * @return string
     */
    public function generateRdf(string $format = '')
    {
        $this->rdf = $this->generateRdfHeader();

        switch ($format) {
            case 'compact':
                $this->generateRdfCompact();
                break;
            case 'nested':
                $this->generateRdfNested();
                break;
            default:
            case 'full':
                $this->generateRdfFull();
                break;
        }


        return $this->rdf . $this->generateRdfFooter();
    }

    /**
     * Genere un format rdf avec  :
     *      <rdf:Description about="ABOUT_URI">
     *          <rdf:type>TYPE</rdf:type>
     *          ...
     */
    protected function generateRdfFull()
    {
        foreach ($this->rdfAssertions as $content) {

            if (empty($content['about']) || empty($content['type'])) {
                continue;
            }
            $this->rdf .= '    <rdf:Description rdf:about="' . $content['about'] . '" >' . "\n";
            $this->rdf .= '        <rdf:type rdf:resource="' . $this->ns['pmb'] . substr($content['type'], 4) . '" />' . "\n";

            foreach ($content['properties'] as $predicate => $objects) {

                if ($predicate == 'pmb:identifier') {
                    continue;
                }

                foreach ($objects as $object) {
                    if ('' !== $object) {
                        $this->rdf .= '        <' . $predicate . '>' . htmlspecialchars($object, ENT_NOQUOTES, 'UTF-8') . '</' . $predicate . '>' . "\n";
                    }
                }
            }
            foreach ($content['resources'] as $predicate => $objects) {
                foreach ($objects as $object) {
                    $this->rdf .= '        <' . $predicate . ' rdf:resource="' . $object . '" />' . "\n";
                }
            }

            $this->rdf .= '    </rdf:Description>' . "\n\n";
        }
    }

    /**
     * Genere un format rdf avec  :
     *      <pmb:TYPE about="ABOUT_URI">
     *          ...
     */
    protected function generateRdfCompact()
    {
        foreach ($this->rdfAssertions as $content) {

            if (empty($content['about']) || empty($content['type'])) {
                continue;
            }
            $ok = true;
            $rdf = '    <' . 'pmb:' . substr($content['type'], 4) . ' rdf:about="' . $content['about'] . '" >' . "\n";
            foreach ($content['properties'] as $predicate => $objects) {

                if ($predicate == 'pmb:identifier') {
                    continue;
                }

                foreach ($objects as $object) {
                    if ('' !== $object) {
                        $rdf .= '        <' . $predicate . '>' . htmlspecialchars($object, ENT_NOQUOTES, 'UTF-8') . '</' . $predicate . '>' . "\n";
                    }
                }
            }
            foreach ($content['resources'] as $predicate => $objects) {
                foreach ($objects as $object) {
                    $rdf .= '        <' . $predicate . ' rdf:resource="' . $object . '" />' . "\n";
                }
            }

            $rdf .= '    </pmb:' . substr($content['type'], 4) . '>' . "\n\n";
            if ($ok) {
                $this->rdf .= $rdf;
            }
        }
    }

    /**
     * Genere un format rdf avec  :
     *      <pmb:TYPE about="ABOUT_URI">
     *          ...
     */
    protected function generateRdfNested()
    {
        while (!empty($this->rdfAssertions)) {

            $rdfAssertion = array_shift($this->rdfAssertions);

            $this->generateRdfNestedAssertion($rdfAssertion);
        }
    }

    protected function generateRdfNestedAssertion(array $rdfAssertion = [])
    {
        if (empty($rdfAssertion['about']) || empty($rdfAssertion['type'])) {
            return;
        }
        $this->rdf .= '    <' . 'pmb:' . substr($rdfAssertion['type'], 4) . ' rdf:about="' . $rdfAssertion['about'] . '" >' . "\n";
        foreach ($rdfAssertion['properties'] as $predicate => $objects) {

            if ($predicate == 'pmb:identifier') {
                continue;
            }

            foreach ($objects as $object) {
                if ('' !== $object) {
                    $this->rdf .= '        <' . $predicate . '>' . htmlspecialchars($object, ENT_NOQUOTES, 'UTF-8') . '</' . $predicate . '>' . "\n";
                }
            }
        }
        foreach ($rdfAssertion['resources'] as $predicate => $objects) {

            foreach ($objects as $object) {
                if (empty($this->rdfAssertions[$object])) {
                    $this->rdf .= '        <' . $predicate . ' rdf:resource="' . $object . '" />' . "\n";
                } else {
                    $this->rdf .= '        <' . $predicate . '>' . "\n";
                    $rdfNestedAssertion = $this->rdfAssertions[$object];
                    unset($this->rdfAssertions[$object]);
                    $this->generateRdfNestedAssertion($rdfNestedAssertion);
                    $this->rdf .= '        </' . $predicate . '>' . "\n";
                }
            }
        }

        $this->rdf .= '    </pmb:' . substr($rdfAssertion['type'], 4) . '>' . "\n\n";
    }

    protected function generateRdfHeader()
    {
        $rdf = '<?xml version="1.0" encoding="UTF-8"?>
            <rdf:RDF' . "\n";
        foreach ($this->ns as $prefix => $namespace) {
            $rdf .= '    xmlns:' . $prefix . '="' . $namespace . '"' . "\n";
        }
        $rdf .= '>' . "\n";
        return $rdf;
    }

    protected function generateRdfFooter()
    {
        return '</rdf:RDF>' . "\n";
    }

    public function setRootEntities(array $entities = [])
    {
        $this->rootEntities = $entities;
    }
}
