<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: HTML_2_PDF.php,v 1.1.6.2.6.3 2025/11/26 08:33:53 jparis Exp $

namespace Pmb\Common\Library\HTML2PDF;

use ReflectionClass;
use Spipu\Html2Pdf\Exception\Html2PdfException;
use Spipu\Html2Pdf\Html2Pdf;
use TCPDF_FONTS;

class HTML_2_PDF extends Html2Pdf
{
    /**
     * Liste des tags autorise
     *
     * @var null|string[]
     */
    protected static $allowed_tags = null;

    /**
     * Liste des fonts a ignorer (fonts icon, fonts incompatibles avec TCPDF, etc.)
     *
     * @var string[]
     */
    protected const IGNORED_FONTS = [
        'fontawesome'
    ];

    public function __construct(
        $orientation = 'P',
        $format = 'A4',
        $lang = 'fr',
        $unicode = true,
        $encoding = 'UTF-8',
        $margins = [5, 5, 5, 8],
        $pdfa = false
    ) {
        parent::__construct($orientation, $format, $lang, $unicode, $encoding, $margins, $pdfa);
        $this->includeFonts();

        // Desactiver la verification des hotes autorises pour permettre le chargement des images
        $this->getSecurityService()->disableCheckAllowedHosts();
    }

    /**
     * Recupere tous les tags autorises
     *
     * @return string[]
     */
    protected function fetchAllowedTags()
    {
        if (isset(static::$allowed_tags)) {
            return static::$allowed_tags;
        }

        // Recuperation des tags geres par les extensions
        static::$allowed_tags = array_map('strtolower', $this->getExtensionTags());

        // Ajout des tags utilises pour la CSS
        // Voir \Spipu\Html2Pdf\Html2Pdf::extractStyle
        static::$allowed_tags[] = 'link';
        static::$allowed_tags[] = 'style';

        // Recuperation des tags geres par HTML2PDF
        $reflection = new ReflectionClass($this);
        foreach ($reflection->getMethods() as $method) {
            $tagname = null;
            if (substr($method->name, 0, 10) == '_tag_open_') {
                $tagname = substr($method->name, 10);
            } elseif (substr($method->name, 0, 11) == '_tag_close_') {
                $tagname = substr($method->name, 11);
            } else {
                continue;
            }

            if (isset($tagname)) {
                static::$allowed_tags[] = strtolower($tagname);
            }
        }

        static::$allowed_tags = array_unique(static::$allowed_tags);
        return static::$allowed_tags;
    }

    /**
     * Retourne tous les tags de l'extension
     *
     * @return string[]
     */
    protected function getExtensionTags()
    {
        if (!$this->extensionsLoaded) {
            $this->loadExtensions();
        }

        return array_keys($this->tagObjects);
    }

    /**
     * Nettoie le HTML en supprimant les balises non autorisees
     *
     * @param string $html
     * @return string
     */
    protected function cleanHTML(string $html): string
    {
        $allowed_tags = $this->fetchAllowedTags();
        $allowed_tags = array_map(function ($tag) {
            return "<{$tag}>";
        }, $allowed_tags);

        return strip_tags(
            $html,
            implode('', $allowed_tags)
        );
    }

    /**
     * convert HTML to PDF
     *
     * @see \Spipu\Html2Pdf\Html2Pdf::writeHTML
     * @param string $html
     * @return HTML_2_PDF
     */
    public function writeHTML($html)
    {
        if (!is_string($html)) {
            throw new Html2PdfException('html must be a string');
        }

        $html = $this->cleanHTML($html);
        return parent::writeHTML($html);
    }

    /**
     * Integre tous les fichiers de police .ttf presents dans ./classes/font et ./styles
     *
     * @param array $dirs
     */
    protected function includeFonts($dirs = null)
    {
        //TODO : integrer les polices .php
        if (is_null($dirs)) {
            global $class_path, $base_path;
            $dirs = [];
            $dirs[] = $class_path . '/font/';
            $dirs[] = $base_path . '/styles/';
        }

        $availableExtensions = ['ttf'];
        foreach ($dirs as $font_path) {
            $dir = dir($font_path);

            while (false !== ($entry = $dir->read())) {

                if (in_array($entry, ['.', '..', 'CVS'])) {
                    continue;
                }

                if (is_dir($font_path . $entry)) {
                    $this->includeFonts([$font_path . $entry . '/']);
                    continue;
                }

                $explode = explode(".", $entry);
                $extension = end($explode);
                if (! in_array($extension, $availableExtensions)) {
                    continue;
                }

                if ($extension == 'ttf') {
                    $this->addTTFfont($font_path . $entry);
                }
            }

            $dir->close();
        }
    }

    /**
     * Generate a temporary folder for fonts
     *
     * @return string
     */
    protected function generateTempFontsFolder()
    {
        global $base_path;

        $tempFontDir = $base_path . '/temp/fonts';
        if (!is_dir($tempFontDir)) {
            mkdir($tempFontDir);
        }

        // Attention, TCPDF veut un chemin absolu avec un / en fin de chaine
        return realpath($tempFontDir) . '/';
    }

    /**
     * Add a TTF font to the TCPDF font list
     *
     * @param string $font_path
     * @return void
     */
    protected function addTTFfont($font_path)
    {
        // Verifier que le fichier existe avant d'essayer de le charger
        if (!file_exists($font_path)) {
            return;
        }

        // Ignorer les fonts icon qui causent des warnings dans TCPDF
        $basename = strtolower(basename($font_path));
        foreach (self::IGNORED_FONTS as $ignoredFont) {
            if (stripos($basename, $ignoredFont) !== false) {
                return;
            }
        }

        TCPDF_FONTS::addTTFfont(
            $font_path,
            'TrueTypeUnicode',
            '',
            32,
            $this->generateTempFontsFolder(),
        );
    }

    /**
     * Return the directories where fonts are cached
     *
     * @return string[]
     */
    public static function cacheDirectories()
    {
        global $base_path;

        return [
            $base_path . '/temp/fonts',
            $base_path . '/opac_css/temp/fonts',
        ];
    }
}
