<?php

// +-------------------------------------------------+
// | 2002-2011 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: html_helper.class.php,v 1.7.4.3 2025/12/31 13:58:59 qvarin Exp $
if (stristr($_SERVER['REQUEST_URI'], ".class.php")) {
    die("no access");
}

class HtmlHelper
{
    public const DEFAULT_COMMON_FOLDER = 'common';

    public const FETCH_CSS = 1;
    public const FETCH_JS = 2;
    public const FETCH_ALL = self::FETCH_CSS | self::FETCH_JS;

    public const STYLE_EXTENSION = '.css';
    public const JS_EXTENSION = '.js';

    /**
     * Instance unique
     *
     * @var HtmlHelper|null
     */
    protected static $instance = null;

    /**
     * Indique si l'instance a ete initialisee
     *
     * @var bool
     */
    protected $is_initialized = false;

    /**
     * Chemin des styles
     *
     * @var string
     */
    protected $styles_path = '';

    /**
     * Contenu des styles (fichiers css et js)
     *
     * @var string|null
     */
    protected $css_content = null;

    /**
     * Fichiers des styles communs desactivees
     *
     * @var array
     */
    protected $common_disabled_files = [];

    /**
     * Constructeur prive
     */
    final private function __construct()
    {}

    /**
     * Retourne une instance
     *
     * @return HtmlHelper
     */
    public static function getInstance()
    {
        if (is_null(static::$instance)) {
            static::$instance = new static();
        }

        static::$instance->getDependencies();
        return static::$instance;
    }

    /**
     * Recupere le contexte d'execution
     *
     * @return void
     */
    protected function getDependencies()
    {
        if (true === $this->is_initialized) {
            return;
        }

        global $base_path;
        $this->styles_path = join(DIRECTORY_SEPARATOR, [$base_path, 'styles']);
        $this->is_initialized = true;
    }

    /**
     * Retourne le dossier des styles
     *
     * @return string
     */
    protected function getStylePath()
    {
        $dir = join(DIRECTORY_SEPARATOR, [__DIR__, '..', 'styles']);
        if ($this->styles_path && is_dir($this->styles_path)) {
            $dir = $this->styles_path;
        }
        return $dir;
    }

    /**
     * Retourne le dossier des styles communs
     *
     * @return string
     */
    public function getCommonFolder()
    {
        global $opac_common_folder;
        $common_path = join(DIRECTORY_SEPARATOR, [$this->getStylePath(), $opac_common_folder]);
        if ($opac_common_folder && is_dir($common_path)) {
            return $opac_common_folder;
        }
        return self::DEFAULT_COMMON_FOLDER;
    }

    /**
     * Lit le fichier disable et ajoute les fichiers dans le tableau $this->common_disabled_files
     *
     * @param string $styles_path
     * @return bool
     */
    protected function fetchDisabledFiles(string $styles_path)
    {
        $disable_path = join(DIRECTORY_SEPARATOR, [$styles_path, $this->getCommonFolder(), 'disable']);

        if (!is_file($disable_path) || !is_readable($disable_path)) {
            return false;
        }

        $s = file_get_contents($disable_path);
        if (empty($s)) {
            return false;
        }

        $files = explode(PHP_EOL, $s);
        foreach ($files as $file) {
            $filename = join(DIRECTORY_SEPARATOR, [$styles_path, $this->getCommonFolder(), trim($file)]);
            if (is_file($filename)) {
                $this->common_disabled_files[] = $filename;
            }
        }
        return true;
    }

    /**
     * Rcupre les fichiers css et js d'un dossier pour les mettres dans $this->css_content
     *
     * @param string $directory
     * @param int $fetch
     * @return void
     */
    protected function fetchFilesFromDirectory(string $directory, int $fetch = self::FETCH_ALL)
    {
        // On s'assure que le dossier termine par un DIRECTORY_SEPARATOR
        $directory = rtrim($directory, '/\\') . DIRECTORY_SEPARATOR;

        // inclusion des feuilles de style
        if ($fetch & self::FETCH_CSS) {
            $css_files = static::readAndSortFiles($directory, self::STYLE_EXTENSION);
            foreach ($css_files as $css_file) {
                $time = @filemtime($css_file);
                $this->css_content .= PHP_EOL . "<link rel='stylesheet' type='text/css' href='" . $this->formatPath($css_file) . "?" . $time . "' />";
            }
        }

        // inclusion des fichiers js
        if ($fetch & self::FETCH_JS) {
            $js_files = static::readAndSortFiles($directory, self::JS_EXTENSION);
            foreach ($js_files as $js_file) {
                $time = @filemtime($js_file);
                $this->css_content .= PHP_EOL . "<script src='" . $this->formatPath($js_file) . "?" . $time . "' ></script>";
            }
        }
    }

    /**
     * Lit les fichiers d'un repertoire en fonction de leur extension et les trie par ordre alpha
     *
     * @param string $dir : repertoire a lire avec / final
     * @param string $extension : extension du fichier
     * @return array|string[] : tableau de fichiers
     */
    protected function readAndSortFiles(string $dir, string $extension)
    {
        global $opac_rgaa_active;

        if (! is_dir($dir)) {
            return [];
        }

        $handle = @opendir($dir);
        if (! $handle) {
            return [];
        }

        $extension_length = strlen($extension);
        $files = [];
        while (false !== ($item = readdir($handle))) {
            if ($extension == self::STYLE_EXTENSION && ! $opac_rgaa_active && strpos($item, 'rgaa') === 0) {
                // Les feuilles de styles avec le prfixe rgaa sont incluses lorsque le parametre est actif
                $this->common_disabled_files[] = $dir . $item;
            }

            if (in_array($dir . $item, $this->common_disabled_files)) {
                // Le fichier est desactive
                continue;
            }

            if (is_file($dir . $item) && substr($item, - $extension_length) == $extension) {
                $files[] = $dir . $item;
            }
        }

        closedir($handle);
        sort($files);

        return $files;
    }

    /**
     * Retourne les feuilles de style et fichiers js associes a inclure
     *
     * @param string $style
     * @return string
     */
    public function getStyle(string $style)
    {
        if (! is_null($this->css_content)) {
            return $this->css_content;
        }

        $directory_style = $this->getStylePath();
        $this->css_content = "";

        // Lecture fichier disable dans le repertoire common
        $this->fetchDisabledFiles($directory_style);

        // inclusion du style/js provenant du dossier common
        $this->fetchFilesFromDirectory(join(DIRECTORY_SEPARATOR, [$directory_style, $this->getCommonFolder()]));

        // inclusion du style/js provenant du dossier style
        if (!empty($style)) {
            $this->fetchFilesFromDirectory(join(DIRECTORY_SEPARATOR, [$directory_style, $style]));
        }

        // Ajout du style dans une globale js
        $this->css_content .= "<script>var opac_style= '" . $style . "';</script>";

        return $this->css_content;
    }

    /**
     * Formate un chemin en remplaant les backslashes par des slashes
     * (Vive Windows...)
     *
     * @param string $path
     * @return string
     */
    protected function formatPath(string $path): string
    {
        return str_replace(["\\", DIRECTORY_SEPARATOR], "/", $path);
    }
}
