<?php
// +-------------------------------------------------+
//  2002-2011 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: list_readers_ai_storage_ui.class.php,v 1.0 2024/11/07

use Pmb\AI\Models\AiSharedListDocnumModel;

if (stristr($_SERVER['REQUEST_URI'], ".class.php")) die("no access");

global $include_path;
require_once($include_path."/templates/list/readers/list_readers_ai_storage_ui.tpl.php");

/**
 * Classe UI pour l'affichage de la liste des lecteurs avec leur espace de stockage IA
 * Hrite de list_readers_ui
 *
 * Fonctionnalits :
 * - Affichage : code-barre, nom, prnom, mail, tl, espace utilis
 * - Filtres : espace min/max (Mo), statut, catgorie
 * - Calcul et affichage de l'espace disque utilis par les sharedlist IA
 */
class list_readers_ai_storage_ui extends list_readers_ui {

	/**
	 * Surcharge de la requte de base pour filtrer les lecteurs avec stockage
	 * Ajoute une JOIN avec opac_liste_lecture et ai_shared_list_docnum
	 * pour exclure les lecteurs sans fichiers uploads
	 *
	 * @return string Requte SQL modifie
	 */
	protected function _get_query_base() {
		$query = 'SELECT DISTINCT id_empr, CONCAT(empr_nom, " ", empr_prenom) AS empr_nom_prenom FROM empr
				JOIN empr_statut ON empr.empr_statut=empr_statut.idstatut
				JOIN empr_categ ON empr.empr_categ=empr_categ.id_categ_empr
				JOIN empr_codestat ON empr.empr_codestat = empr_codestat.idcode
                JOIN docs_location ON empr.empr_location=docs_location.idlocation
                LEFT JOIN opac_liste_lecture oll ON oll.num_empr = empr.id_empr
                LEFT JOIN ai_shared_list_docnum asd ON asd.num_list_ai_shared_list_docnum = oll.id_liste
                WHERE asd.id_ai_shared_list_docnum IS NOT NULL';
		return $query;
	}


	/**
	 * Initialisation des filtres slectionns par dfaut
	 * Affiche les filtres pertinents pour cette page
	 */
	protected function init_default_selected_filters() {
		global $pmb_lecteurs_localises;

		// Ajout du filtre de localisation si applicable
		if($pmb_lecteurs_localises) {
			$this->add_selected_filter('location');
		}

		// Filtre de statut du lecteur
		$this->add_selected_filter('status');
		$this->add_empty_selected_filter();

		// Filtre de catgorie du lecteur
		$this->add_selected_filter('categorie');
	}

	/**
	 * Initialisation des colonnes disponibles
	 * Ajoute la colonne personnalise "storage_used"
	 */
	protected function init_available_columns() {
		parent::init_available_columns();

		// Ajout de la colonne d'espace utilis
		$this->available_columns['main_fields']['storage_used'] = 'edit_empr_storage_used';
	}

	/**
	 * Initialisation des colonnes affiches par dfaut
	 * Affiche uniquement les colonnes demandes :
	 * code-barre, nom, prnom (dans empr_name), mail, tl, espace utilis
	 */
	protected function init_default_columns() {
		// Ajout de la colonne de slection si des actions de slection sont disponibles
		$selection_actions = $this->get_selection_actions();
		if(is_countable($selection_actions) && count($selection_actions)) {
			$this->add_column_selection();
		}

		// Colonnes affiches
		$this->add_column('cb');        // Code-barre
		$this->add_column('empr_name'); // Nom et prnom
		$this->add_column('mail');      // Email
		$this->add_column('tel1');      // Tl
		$this->add_column('storage_used'); // Espace utilis (colonne personnalise)
	}

	/**
	 * Initialisation des actions de slection par dfaut
	 * Ajoute l'action "Supprimer les fichiers" pour suppression en masse
	 */
	protected function init_default_selection_actions() {
		global $msg;

		parent::init_default_selection_actions();

		// Ajout de l'action "Supprimer les fichiers IA" (sans icne)
		$link = array(
			'href' => './edit.php?categ=empr&sub=ai_storage&action=delete_storage_files',
			'confirm' => $msg['edit_empr_delete_storage_files_confirm']
		);
		$this->add_selection_action('delete_storage_files', $msg['edit_empr_delete_storage_files'], '', $link);
	}

	/**
	 * Rcupre le sous-titre de la page
	 * Affich dans le titre de la page
	 *
	 * @return string Sous-titre en franais
	 */
	protected function get_sub_title() {
		// Retourne le titre franais pour cette page
		return "Espace de stockage des lecteurs";
	}

	/**
	 * Rcupre le titre HTML complet de la page
	 *
	 * @return string Titre HTML format
	 */
	protected function get_html_title() {
		return "<h1>Lecteurs: Espace de stockage des lecteurs</h1>";
	}

	/**
	 * Titre du tableur Excel (export)
	 * Affich dans le document export
	 */
	protected function get_display_spreadsheet_title() {
		$this->spreadsheet->write_string(0,0,"Lecteurs: Espace de stockage des lecteurs");
	}

	/**
	 * Rcupre le contenu d'une cellule du tableau
	 * Traite les colonnes personnalises
	 *
	 * @param object $object Objet lecteur (instance de emprunteur)
	 * @param string $property Nom de la colonne
	 * @return string Contenu format de la cellule
	 */
	protected function get_cell_content($object, $property) {
		switch($property) {
			case 'storage_used':
				return $this->_get_object_property_storage_used($object);
			default:
				return parent::get_cell_content($object, $property);
		}
	}

	/**
	 * Calcule et formate l'espace de stockage utilis par un lecteur
	 * Rcupre les listes partages du lecteur et calcule la taille totale des fichiers
	 *
	 * @param object $object Objet lecteur (instance de emprunteur)
	 * @return string Espace format en Ko/Mo/Go
	 */
	protected function _get_object_property_storage_used($object) {
		$total_size = $this->calculate_storage($object->id);
		return $this->format_storage_size($total_size);
	}

	/**
	 * Calcule l'espace de stockage utilis par un lecteur
	 * Parcourt toutes les listes partages (opac_liste_lecture) du lecteur
	 * et calcule la taille totale des fichiers (ai_shared_list_docnum) associs
	 *
	 * @param int $id_empr ID du lecteur
	 * @return int Taille totale en octets
	 */
	protected function calculate_storage($id_empr) {
		return AiSharedListDocnumModel::calculate_storage($id_empr);
	}

	/**
	 * Formate une taille en octets en format lisible (Ko, Mo, Go)
	 * Affiche 2 dcimales
	 *
	 * @param int $bytes Taille en octets
	 * @return string Taille formate (ex: "2.50 Mo")
	 */
	protected function format_storage_size($bytes) {
		$bytes = intval($bytes);

		// Conversion en Go
		if ($bytes >= 1073741824) {
			return number_format($bytes / 1073741824, 2, '.', ' ') . ' Go';
		}
		// Conversion en Mo
		elseif ($bytes >= 1048576) {
			return number_format($bytes / 1048576, 2, '.', ' ') . ' Mo';
		}
		// Conversion en Ko
		elseif ($bytes >= 1024) {
			return number_format($bytes / 1024, 2, '.', ' ') . ' Ko';
		}
		// Affichage en octets
		else {
			return $bytes . ' octets';
		}
	}

}
