<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: DashboardController.php,v 1.21.8.3 2025/08/29 09:38:36 jparis Exp $
namespace Pmb\Dashboard\Controller;

use Pmb\Common\Controller\Controller;
use Pmb\Common\Helper\Helper;
use Pmb\Common\Orm\UsersGroupsOrm;
use Pmb\Dashboard\Models\DashboardModel;
use Pmb\Dashboard\Models\DashboardUsersGroupsModel;
use Pmb\Dashboard\Models\DashboardWidgetModel;
use Pmb\Dashboard\Models\WidgetModel;
use Pmb\Dashboard\Views\DashboardView;

class DashboardController extends Controller
{
    /**
     * Affiche la vue du tableau de bord.
     *
     * @return void
     */
    public function proceed()
    {
        $data = $this->getDataView();

        $view = new DashboardView("dashboard/dashboard", $data);
        print $view->render();
    }

    /**
     * Rcupre les donnes pour la vue
     *
     * @return array
     */
    private function getDataView()
    {
        global $msg, $PMBuserid, $pmb_url_base;

        $data = array();

        $widget = new WidgetModel();
        $data["widgets"] = Helper::toArray($widget->getListByCurrentUserId());
        $data["dashboards"] = $this->getFormatedDashboardList();

        $data["defaultDashboard"] = DashboardModel::fetchUserDefaultDashboard();

        $data["groups"] = [
            ["grp_id" => -1, "grp_name" => $msg["form_share_all_dashboard"]],
            ["grp_id" => 0, "grp_name" => $msg["admin_usr_grp_non_aff"]]
        ];

        $data["users"] = $this->getUserList();

        $usersOrm = UsersGroupsOrm::findAll("grp_name");

        for ($index = 0; $index < count($usersOrm); $index++) {
            $data["groups"][] = ["grp_id" => $usersOrm[$index]->grp_id, "grp_name" => $usersOrm[$index]->grp_name];
        }

        $data["current_user"] = intval($PMBuserid);
        $data["current_group"] = intval(\user::get_param($data["current_user"], "grp_num"));

        $data["widget_types"] = WidgetModel::getWidgetTypeList();

        $data["url_webservice"] = $pmb_url_base . "rest.php/dashboard/";
        $data["pmb_url_base"] = $pmb_url_base;
        $data["widget_refresh_time"] = 60;

        return $data;
    }

    public function getUserList() 
    {
        $users = [];

        $query = "SELECT userid, username FROM users";
        $result = pmb_mysql_query($query);

        while ($row = pmb_mysql_fetch_assoc($result)) {
            $users[$row["userid"]] = $row["username"];
        }

        return $users;
    }

    public function getFormatedDashboardList()
    {
        $dashboard = new DashboardModel();
        $dashboards = Helper::toArray($dashboard->getListByCurrentUserId());

        $dashboardUsersGroups = new DashboardUsersGroupsModel();
        $dashboardWidget = new DashboardWidgetModel();

        foreach ($dashboards as &$element) {

            if (!isset($element["layout"])) {
                $element["layout"] = [];
            }
            if (!isset($element["dashboardUsersGroups"])) {
                $element["dashboardUsersGroups"] = [];
            }
            if (!isset($element["widgets"])) {
                $element["widgets"] = [];
            }

            $tempDashboardUsersGroups = $dashboardUsersGroups->getList(["num_dashboard" => $element["idDashboard"]], true);

            foreach ($tempDashboardUsersGroups as $value) {
                $element["dashboardUsersGroups"][] = $value["numUsersGroups"];
            }

            $tempDashboardWidget = $dashboardWidget->getList(["num_dashboard" => $element["idDashboard"]], true);

            foreach ($tempDashboardWidget as $value) {
                $element["layout"][] = $value["dashboardWidgetSettings"]["position"];

                $tempWidget = new WidgetModel($value["dashboardWidgetSettings"]["position"]["i"]);

                $dashboardWidgetSettings = ["dashboardWidgetSettings" => $value["dashboardWidgetSettings"]];
                $element["widgets"][] = array_merge(Helper::toArray($tempWidget), $dashboardWidgetSettings);
            }
        }

        return $dashboards;
    }

    /**
     * Rcupre la liste des dashboards formats
     *
     * @return void
     */
    public function getList()
    {
        $this->ajaxJsonResponse($this->getFormatedDashboardList());
    }

    /**
     * Sauvegarde les donnes et renvoie une rponse JSON.
     *
     * @return void
     */
    public function save()
    {
        $this->data->idDashboard = intval($this->data->idDashboard);

        $dashboard = new DashboardModel($this->data->idDashboard);

        $check = $dashboard->check($this->data);


        if ($check['error']) {
            $this->ajaxError($check['errorMessage']);
            exit();
        }

        $dashboard->setFromForm($this->data);

        if ($dashboard->idDashboard) {
            $dashboard->update();
        } else {
            $dashboard->create();
        }

        $dashboardUsersGroups = new DashboardUsersGroupsModel();
        $dashboardUsersGroups->numDashboard = $dashboard->idDashboard;

        $dashboardUsersGroups->deleteByIdDashboard();

        if (in_array(-1, $this->data->dashboardUsersGroups)) {
            $this->data->dashboardUsersGroups = [-1];
        }

        foreach ($this->data->dashboardUsersGroups as $usersGroups) {
            $dashboardUsersGroups = new DashboardUsersGroupsModel();

            $dashboardUsersGroups->numDashboard = $dashboard->idDashboard;
            $dashboardUsersGroups->numUsersGroups = intval($usersGroups);

            $dashboardUsersGroups->create();
        }

        $this->ajaxJsonResponse($dashboard);
    }

    /**
     * Supprime un dashboard de la base de donnes.
     *
     * @return void
     */
    public function delete()
    {
        $this->data->idDashboard = intval($this->data->idDashboard);

        $dashboard = new DashboardModel($this->data->idDashboard);
        $result = $dashboard->delete();

        if ($result['error']) {
            $this->ajaxError($result['errorMessage']);
            exit();
        }
        $this->ajaxJsonResponse([
            'success' => true
        ]);
    }

    /**
     * Duplique un dashboard de la base de donnes.
     *
     * @return void
     */
    public function duplicate()
    {
        global $PMBuserid;

        $this->data->idDashboard = intval($this->data->idDashboard);

        // On duplique le dashboard
        $dashboardModel = new DashboardModel($this->data->idDashboard);
        $dashboardModel->idDashboard = 0;
        $dashboardModel->numUser = intval($PMBuserid);
        $dashboardModel->dashboardName = "Copy of " . $dashboardModel->dashboardName;

        $dashboardModel->create();

        // On duplique les liens entre le dashboard et les widgets
        $dashboardWidgetModel = new DashboardWidgetModel();
        $dashboardWidgetList = $dashboardWidgetModel->getList(["num_dashboard" => $this->data->idDashboard], false);

        foreach($dashboardWidgetList as $dashboardWidget) {
            $dashboardWidget->numDashboard = $dashboardModel->idDashboard;
            $dashboardWidget->dashboardWidgetSettings = json_encode($dashboardWidget->dashboardWidgetSettings);
            $dashboardWidget->update();
        }

        // On duplique les liens entre le dashboard et les groupes d'utilisateurs
        $dashboardUserGroupModel = new DashboardUsersGroupsModel();
        $dashboardUserGroupList = $dashboardUserGroupModel->getList(["num_dashboard" => $this->data->idDashboard], false);

        foreach($dashboardUserGroupList as $dashboardUserGroup) {
            $dashboardUserGroup->numDashboard = $dashboardModel->idDashboard;
            $dashboardUserGroup->create();
        }

        $this->ajaxJsonResponse([
            'success' => true
        ]);
    }

    /**
     * Sauvegarde le layout du tableau de bord.
     *
     * @return void
     */
    public function saveLayout()
    {
        global $PMBuserid;

        $this->data->idDashboard = intval($this->data->idDashboard);

        $dashboard = new DashboardModel($this->data->idDashboard);

        // Si on est pas propritaire du dashboard et qu'il n'est pas editable, on renvoi une erreur
        if ($dashboard->numUser != $PMBuserid && !$dashboard->dashboardEditable) {
            $this->ajaxError("msg:form_not_allowed");
            exit();
        }

        $dashboardWidget = new DashboardWidgetModel();
        $dashboardWidget->numDashboard = $this->data->idDashboard;
        $dashboardWidget->deleteByIdDashboard();

        foreach ($this->data->layout as $value) {
            $dashboardWidget->numDashboard = $this->data->idDashboard;
            $dashboardWidget->numWidget = $value->i;

            foreach ($this->data->dashboardWidgets as $element) {
                if ($element->numWidget == $value->i) {
                    $dashboardWidget->dashboardWidgetSettings = $element->dashboardWidgetSettings;
                    $dashboardWidget->dashboardWidgetSettings->position = $value;
                    $dashboardWidget->dashboardWidgetSettings = json_encode($dashboardWidget->dashboardWidgetSettings, true);
                    break;
                }
            }

            $dashboardWidget->update();
        }

        $this->ajaxJsonResponse([
            'success' => true
        ]);
    }

    public function saveDashboardWidget()
    {
        global $PMBuserid;

        $this->data->idDashboard = intval($this->data->idDashboard);

        $dashboard = new DashboardModel($this->data->idDashboard);

        // Si on est pas propritaire du dashboard et qu'il n'est pas editable, on renvoi une erreur
        if ($dashboard->numUser != $PMBuserid && !$dashboard->dashboardEditable) {
            $this->ajaxError("msg:form_not_allowed");
            exit();
        }

        $widget = new WidgetModel(intval($this->data->widget->idWidget));

        // Si on est propritaire ou que le widget est editable alors on peut modifier le contenu
        if ($widget->numUser == $PMBuserid || $widget->widgetEditable) {
            $widget->setFromForm($this->data->widget);

            if ($widget->idWidget) {
                $widget->update();
            } else {
                $widget->create();
            }
        }

        $dashboardWidget = new DashboardWidgetModel();
        $dashboardWidget->numDashboard = $this->data->idDashboard;
        $dashboardWidget->numWidget = $this->data->widget->idWidget;
        $dashboardWidget->dashboardWidgetSettings = $this->data->widget->dashboardWidgetSettings;
        $dashboardWidget->dashboardWidgetSettings = json_encode($dashboardWidget->dashboardWidgetSettings, true);

        $dashboardWidget->update();

        $this->ajaxJsonResponse([
            'success' => true
        ]);
    }

    public function refreshWidget()
    {
        $this->data->idWidget = intval($this->data->idWidget);
        $this->data->idDashboard = intval($this->data->idDashboard);

        $widget = new WidgetModel($this->data->idWidget);
        $dashboardWidget = new DashboardWidgetModel($this->data->idDashboard, $this->data->idWidget);

        if (empty($dashboardWidget->dashboardWidgetSettings)) {
            $this->ajaxJsonResponse([
                'error' => true,
                'errorMessage' => '',
            ]);
            exit();
        }

        $this->ajaxJsonResponse(
            array_merge(
                Helper::toArray($widget),
                ["dashboardWidgetSettings" => $dashboardWidget->dashboardWidgetSettings]
            )
        );
    }

    /**
     * Exporte un dashboard
     *
     * @return void
     */
    public function export()
    {
        global $PMBuserid;

        $idDashboard = intval($this->data->idDashboard);
        if (!$idDashboard) {
            http_response_code(400);
            print json_encode(['error' => true, 'errorMessage' => 'msg:form_data_errors']);
            return;
        }

        $dashboard = new DashboardModel($idDashboard);

        $allowed = false;
        $usersGroupId = intval(\user::get_param(intval($PMBuserid), 'grp_num'));
        $dguModel = new DashboardUsersGroupsModel();
        $links = $dguModel->getList(["num_dashboard" => $idDashboard], true);
        foreach ($links as $lg) {
            if (intval($lg['numUsersGroups']) === -1 || intval($lg['numUsersGroups'] === $usersGroupId)) {
                $allowed = true;
                break;
            }
        }

        if (!$allowed) {
            http_response_code(403);
            print json_encode(['error' => true, 'errorMessage' => 'msg:form_not_allowed']);
            return;
        }

        $export = [];
        $export['version'] = 1;
        $export['dashboard'] = [
            'dashboardName' => $dashboard->dashboardName,
            'dashboardEditable' => intval($dashboard->dashboardEditable),
        ];

        $dgu = new DashboardUsersGroupsModel();
        $dguList = $dgu->getList(["num_dashboard" => $idDashboard], true);
        $export['dashboardUsersGroups'] = [];
        foreach ($dguList as $row) {
            $export['dashboardUsersGroups'][] = intval($row['numUsersGroups']);
        }

        $dw = new DashboardWidgetModel();
        $dwList = $dw->getList(["num_dashboard" => $idDashboard], true);
        $export['widgets'] = [];
        $export['procs'] = [];

        foreach ($dwList as $row) {
            $w = new WidgetModel(intval($row['numWidget']));
            $export['widgets'][] = array_merge(
                Helper::toArray($w),
                ['dashboardWidgetSettings' => $row['dashboardWidgetSettings']]
            );

            // Verifie si le widget est de type 'stat' sinon on doit exporter les procedures utilisees
            if ($w->widgetType === 'stat') {
                $methods = is_array($w->widgetSettings["methods"]) ? $w->widgetSettings["methods"] : [];

                foreach ($methods as $method) {
                    if (
                        isset($method["module"], $method["id"]) &&
                        $method["module"] === 'proc' &&
                        ($idProc = intval($method["id"])) > 0
                    ) {
                        $query = "SELECT idproc, name, requete, comment, parameters, proc_notice_tpl, proc_notice_tpl_field 
                                FROM procs 
                                WHERE idproc = $idProc";
                        $result = pmb_mysql_query($query);

                        if ($result && pmb_mysql_num_rows($result)) {
                            $row = pmb_mysql_fetch_object($result);
                            $export['procs'][] = Helper::toArray($row);
                        }
                    }
                }
            }
        }

        $filename = 'dashboard_' . $dashboard->dashboardName . '.json';
        header('Content-Type: application/json; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        echo json_encode($export);
    }

    /**
     * Importe un dashboard
     *
     * @return void
     */
    public function import()
    {
        global $PMBuserid;

        if (empty($this->data) || empty($this->data->dump)) {
            $this->ajaxError('msg:form_data_errors');
            return;
        }

        $dump = $this->data->dump;
        if (is_string($dump)) {
            $dump = json_decode($dump, true);
        }

        if (!is_array($dump) || empty($dump['dashboard']) || !is_array(($dump['widgets'] ?? null))) {
            $this->ajaxError('msg:form_data_errors');
            return;
        }

        $dashboard = new DashboardModel();
        $dashboard->dashboardName = $dump['dashboard']['dashboardName'] ?? 'Imported dashboard';
        $dashboard->dashboardEditable = intval($dump['dashboard']['dashboardEditable'] ?? 0);
        $dashboard->numUser = intval($PMBuserid);
        $dashboard->create();

        foreach ($dump['widgets'] as $w) {
            $new = new WidgetModel();
            $new->widgetName = $w['widgetName'] ?? 'Widget';
            $new->widgetEditable = intval($w['widgetEditable'] ?? 0);
            $new->widgetType = $w['widgetType'] ?? '';
            $new->numUser = intval($PMBuserid);
            $new->widgetShareable = intval($w['widgetShareable'] ?? 0);
            $new->widgetSettings = $w['widgetSettings'] ?? [];

            $settings = $w['dashboardWidgetSettings'] ?? [];

            // Specifique widget Compteur: forcer location avec $deflt_docs_location si absent
            if (strtolower($new->widgetType) === 'counter') {
                global $deflt_docs_location;
                if (!is_array($new->widgetSettings)) {
                    $new->widgetSettings = [];
                }

                // Forcer systematiquement la localisation
                $location = intval($deflt_docs_location);
                $new->widgetSettings['location'] = $location;
                $settings['location'] = $location;

                // Forcer systematiquement la reinitialisation des statistiques de visites
                unset($new->widgetSettings['counters']);
                unset($settings['counters']);
            }

            // Secifique widget stats on import aussi la procdure dans un classement 'Dashboard'
            if (strtolower($new->widgetType) === 'stat') {
                $methods = is_array($settings["methods"]) ? $settings["methods"] : [];

                foreach ($methods as &$method) {
                    if (
                        isset($method["module"], $method["id"]) &&
                        $method["module"] === 'proc' &&
                        ($idProc = intval($method["id"])) > 0
                    ) {
                        foreach ($dump['procs'] as $proc) {
                            if ($proc['idproc'] == $idProc) {

                                // Cration d'un classement 'Dashboard' ou rcupration si existant
                                $procClassment = new \procs_classement();
                                $procClassmentId = $procClassment::get_id_from_libelle("Dashboard");
                                if($procClassmentId == 0) {
                                    $procClassment->libelle = "Dashboard";
                                    $procClassment->save();
                                    $procClassmentId = $procClassment->id;
                                }

                                $query = sprintf(
                                    'INSERT INTO procs (
                                        name, requete, comment, autorisations, autorisations_all,
                                        parameters, num_classement, proc_notice_tpl, proc_notice_tpl_field
                                    ) VALUES (
                                        "%s", "%s", "%s", "%s", %s, "%s", %s, "%s", "%s"
                                    )',
                                    addslashes($proc['name']),
                                    addslashes($proc['requete']),
                                    addslashes($proc['comment']),
                                    '1',
                                    '1',
                                    addslashes($proc['parameters']),
                                    $procClassmentId,
                                    addslashes($proc['proc_notice_tpl']),
                                    addslashes($proc['proc_notice_tpl_field'])
                                );
                                pmb_mysql_query($query);
                                $method['id'] = pmb_mysql_insert_id();

                                if(isset($method['conditions']) && is_array($method['conditions'])) {
                                    foreach ($method['conditions'] as &$condition) {
                                        // Remettre a vide selon le type de donnes
                                        if(is_int($condition)) {
                                            $condition = 0;
                                        } elseif(is_array($condition)) {
                                            $condition = [];
                                        } elseif(is_string($condition)) {
                                            $condition = '';
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                $new->widgetSettings["methods"] = $methods;
                $settings['methods'] = $methods;
            }

            $new->create();

            if (isset($settings['position'])) {
                $settings['position']['i'] = intval($new->idWidget);
            }

            $link = new DashboardWidgetModel();
            $link->numDashboard = $dashboard->idDashboard;
            $link->numWidget = $new->idWidget;
            $link->dashboardWidgetSettings = json_encode($settings, true);
            $link->update();
        }

        // On force le groupe non affecte
        $dgu = new DashboardUsersGroupsModel();
        $dgu->numDashboard = $dashboard->idDashboard;
        $dgu->numUsersGroups = 0;
        $dgu->create();

        $this->ajaxJsonResponse([
            'success' => true,
            'idDashboard' => $dashboard->idDashboard,
        ]);
    }
}
