<?php
// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: OcrServer.php,v 1.1.2.1 2025/10/21 08:22:46 qvarin Exp $

namespace Pmb\Ocr\Library\Server;

use Curl;
use CURLFile;
use CurlResponse;
use encoding_normalize;

class OcrServer
{
    public const STATE_PENDING = 1;
    public const STATE_IN_PROGRESS = 2;
    public const STATE_COMPLETED = 3;
    public const STATE_UNKNOWN = 0;

    /**
     * Endpoints pour rcuperer l'tat, du pdf envoy
     *
     * @var string
     */
    public const OCR_STATE_ENDPOINT = 'ocr/state';

    /**
     * Endpoints pour envoyer le pdf
     *
     * @var string
     */
    public const OCR_PUSH_ENDPOINT = 'ocr/push';

    /**
     * Endpoints pour recevoir le pdf
     *
     * @var string
     */
    public const OCR_PULL_ENDPOINT = 'ocr/pull';

    /**
     * Is the server active
     *
     * @var bool
     */
    private $active = false;

    /**
     * Configuration server
     *
     * @var array|null
     */
    protected $configServer = null;

    public function __construct()
    {
        $this->getConfigServer();
    }

    /**
     * Check if the server is active
     *
     * @return boolean
     */
    public function isActive(): bool
    {
        global $pmb_ocr_active;

        if (!$pmb_ocr_active || empty($this->configServer['service'])) {
            return false;
        }
        return $this->configServer['service'] == 2;
    }

    /**
     * Get the config server
     *
     * @return array
     */
    public function getConfigServer(): array
    {
        if ($this->configServer === null) {
            global $pmb_ocr_config;
            $this->configServer = encoding_normalize::json_decode($pmb_ocr_config, true) ?? [];
        }

        if (empty($this->configServer) || empty($this->configServer['url']) || empty($this->configServer['salt']) || empty($this->configServer['token'])) {
            $this->active = false;
        }

        return $this->configServer;
    }

    /**
     * Get the config server
     *
     * @param string $config
     * @param mixed $default
     * @return mixed
     */
    public function get(string $config, $default = null): mixed
    {
        return $this->getConfigServer()[$config] ?? $default;
    }

    /**
     * Get the headers
     *
     * @return array
     */
    protected function getHeaders(): array
    {
        return [
            'X-Requested-With' => 'XMLHttpRequest',
            'X-Ocr-Server-Salt' => $this->get('salt'),
            'X-Ocr-Server-Token' => $this->get('token'),
        ];
    }

    /**
     * Build the url
     *
     * @param string $endpoint
     * @return string
     */
    protected function buildUrl(string $endpoint): string
    {
        $url = rtrim($this->get('url', ''), '/');
        return $url . '/' . $endpoint;
    }

    /**
     * Get the state
     *
     * @param string $hash
     * @return false|array
     */
    public function state(string $hash)
    {
        if ($this->getConfigServer() && !$this->isActive()) {
            return false;
        }

        $curl = new Curl();
        $curl->headers = $this->getHeaders();
        $curlResponse = $curl->post($this->buildUrl(self::OCR_STATE_ENDPOINT), ['hash' => $hash]);

        if ($curlResponse instanceof CurlResponse) {
            switch ($curlResponse->headers['Status-Code']) {
                case '200':
                    $data = encoding_normalize::json_decode($curlResponse->body, true);
                    return ['success' => true, 'data' => $data];

                default:
                    return [
                        'success' => false,
                        'code' => $curlResponse->headers['Status-Code'],
                        'message' => $curlResponse->headers['X-Error-Message'] ?? $curlResponse->headers['Status'],
                    ];
            }
        }

        return false;
    }

    /**
     * Push the data
     *
     * @param string $filename
     * @return false|array
     */
    public function push(string $filename)
    {
        if ($this->getConfigServer() && !$this->isActive()) {
            return false;
        }
        $curl = new Curl();
        $curl->headers = $this->getHeaders();
        $curl->options['CURLOPT_POSTFIELDS'] = [
            'ocr_file' => new CURLFile($filename),
        ];

        $curlResponse = $curl->post($this->buildUrl(self::OCR_PUSH_ENDPOINT));
        if ($curlResponse instanceof CurlResponse) {
            switch ($curlResponse->headers['Status-Code']) {
                case '200':
                    $data = encoding_normalize::json_decode($curlResponse->body, true);
                    return ['success' => true, 'data' => $data];

                default:
                    return [
                        'success' => false,
                        'code' => $curlResponse->headers['Status-Code'],
                        'message' => $curlResponse->headers['X-Error-Message'] ?? $curlResponse->headers['Status'],
                    ];
            }
        }

        return false;
    }

    /**
     * Pull the data
     *
     * @param string $hash
     * @return mixed
     */
    public function pull(string $hash)
    {
        if ($this->getConfigServer() && !$this->isActive()) {
            return false;
        }

        $curl = new Curl();
        $curl->headers = $this->getHeaders();
        $curlResponse = $curl->post($this->buildUrl(self::OCR_PULL_ENDPOINT), ['hash' => $hash]);
        if ($curlResponse instanceof CurlResponse) {
            switch ($curlResponse->headers['Status-Code']) {
                case '200':
                    if ($curlResponse->headers['Content-Type'] == 'application/pdf') {
                        return ['success' => true, 'data' => $curlResponse->body];
                    }
                    return ['success' => false, 'data' => null];

                case '204':
                    // Fichier non traite
                    return ['success' => false, 'data' => null];

                default:
                    return [
                        'success' => false,
                        'code' => $curlResponse->headers['Status-Code'],
                        'message' => $curlResponse->headers['X-Error-Message'] ?? $curlResponse->headers['Status'],
                    ];
            }
        }

        return false;
    }
}
