class Calendar {
    container = "";

    mode = "datePicker";
    data = [];

    _language = "";
    currentDate = new Date();


    constructor(containerId, mode = "datePicker", data = []) {
        this.id = `calendar-${Math.random().toString(36).slice(2, 11)}`;

        this.setContainer(containerId);
        this.setMode(mode, false);
        this.setData(data, false);

        this.buildRender();
    }

    /**
     * Affiche le calendrier.
     *
     * @returns {void}
     */
    buildRender() {
        this.container.innerHTML = "";
        this.createCalendar();
        this.render();
    }

    /**
     * Affiche le calendrier.
     *
     * @returns {void}
     */
    render() {
        this.renderCalendar();
        this.setupEvents();
    }

    /**
     * Crée le contenu HTML du calendrier et le place dans le conteneur.
     *
     * @returns {void}
     */
    createCalendar() {
        const weekdays = this.getWeekdays();
        const months = this.getMonths();
        const years = this.getYears();

        const monthsOptions = months.map((month, index) => `<option value="${index}" ${index === this.currentDate.getMonth() ? "selected" : ""}>${month}</option>`);
        const yearsOptions = years.map((year) => `<option value="${year}" ${year === this.currentDate.getFullYear() ? "selected" : ""}>${year}</option>`);
        const weekDaysHtml = weekdays.map((day) => `<li><abbr title="${day.label}">${day.abbr}</abbr></li>`);

        const date = new Date(this.currentDate);
        const previousMonth = date.setMonth(date.getMonth() - 1);
        const nextMonth = date.setMonth(date.getMonth() + 2);

        const longFormatter = new Intl.DateTimeFormat(this.language, {
            month: "long",
            year: "numeric",
        });

        const previousMonthLabel = pmbDojo.messages.getMessage("calendar", "previous_month").replace("!!previous_month!!", longFormatter.format(previousMonth));
        const nextMonthLabel = pmbDojo.messages.getMessage("calendar", "next_month").replace("!!next_month!!", longFormatter.format(nextMonth));
        const selectMonthLabel = pmbDojo.messages.getMessage("calendar", "select_month");
        const selectYearLabel = pmbDojo.messages.getMessage("calendar", "select_year");

        const currentMonthLabel = longFormatter.format(this.currentDate);
        const currentMonthDatetime = `${this.currentDate.getFullYear()}-${this.pad(this.currentDate.getMonth() + 1)}`;

        this.container.innerHTML = `
            <div id="${this.id}" class="calendar">
                <div class="calendar-header">
                    <button class="prev-month" aria-label="${previousMonthLabel}" title="${previousMonthLabel}" tabindex="0">
                        <img src="${pmbDojo.images.getImage('previous1.png')}" alt="">
                    </button>
                    <select class="month-select" aria-label="${selectMonthLabel}" title="${selectMonthLabel}">${monthsOptions.join("")}</select>
                    <select class="year-select" aria-label="${selectYearLabel}" title="${selectYearLabel}">${yearsOptions.join("")}</select>
                    <button class="next-month" aria-label="${nextMonthLabel}" title="${nextMonthLabel}" tabindex="0">
                        <img src="${pmbDojo.images.getImage('next1.png')}" alt="">
                    </button>
                </div>
                <ul class="calendar-weekdays">${weekDaysHtml.join("")}</ul>
                <div class="calendar-body">
                    <time class="month-year" datetime="${currentMonthDatetime}" aria-label="${currentMonthLabel}"></time>
                    <div class="calendar-days"></div>
                </div>
            </div>
        `;
    }

    /**
     * Génère le HTML des jours du mois.
     *
     * @returns {string} Le HTML des jours du mois.
     */
    generateDays() {
        let daysHtml = "";

        const today = new Date();

        // Premier jour du mois courant
        const firstDayOfMonth = new Date(this.currentDate.getFullYear(), this.currentDate.getMonth(), 1);

        // Trouver le lundi de la semaine contenant le 1er
        const startDay = new Date(firstDayOfMonth);
        const dayOfWeek = startDay.getDay();
        const diffToMonday = (dayOfWeek + 6) % 7; // transforme dimanche (0) en 6, lundi (1) en 0, etc.
        startDay.setDate(startDay.getDate() - diffToMonday);

        for (let i = 0; i < 35; i++) {
            const currentDay = new Date(startDay);
            currentDay.setDate(startDay.getDate() + i);

            const day = currentDay.getDate();
            const month = currentDay.getMonth() + 1;
            const year = currentDay.getFullYear();

            const isToday = currentDay.toDateString() === today.toDateString();
            const isWeekend = currentDay.getDay() === 0 || currentDay.getDay() === 6;
            const isOutsideCurrentMonth = currentDay.getMonth() !== this.currentDate.getMonth();

            const weekendAttr = isWeekend ? "data-weekend" : "";

            let timeClass = "";
            if (isToday) {
                timeClass = "selected";
            }
            if (isOutsideCurrentMonth) {
                timeClass += (timeClass ? " " : "") + "outside-month";
            }

            const weekNumber = this.getWeekNumber(currentDay);

            let weekNumberHtml = "";
            if (currentDay.getDay() === 1 || i === 0) {
                const weekNumberLabel = pmbDojo.messages.getMessage("calendar", "week").replace("!!week!!", weekNumber);
                weekNumberHtml = `<abbr title="${weekNumberLabel}" class="weeknumber">${weekNumber}</abbr>`;
            }

            const datetime = `${year}-${this.pad(month)}-${this.pad(day)}`;
            const { eventsHtml, nbEvents } = this.computeEvents(datetime);

            const longFormatter = new Intl.DateTimeFormat(this.language, {
                weekday: "long",
                day: "numeric",
                month: "long",
                year: "numeric",
            });

            const uniqueId = `${this.id}-day-${year}${this.pad(month)}${this.pad(day)}`;
            const dayTitle = `<time class="day-title" datetime="${datetime}">
                <span aria-label="${longFormatter.format(currentDay)}">${this.pad(day)}</span> ${nbEvents}
            </time>`;

            if (isOutsideCurrentMonth) {
                daysHtml += `
                    <li data-day="${day}" data-weeknumber="${weekNumber}" ${weekendAttr} class="day${timeClass ? " " + timeClass : ""}">
                        ${weekNumberHtml}
                        <div class="day-button outside-month-button" id="${uniqueId}">
                            ${dayTitle}
                            ${eventsHtml}
                        </div>
                    </li>
                `;
            } else {
                daysHtml += `
                    <li data-day="${day}" data-weeknumber="${weekNumber}" ${weekendAttr} class="day${timeClass ? " " + timeClass : ""}">
                        ${weekNumberHtml}
                        <button type="button" class="day-button" id="${uniqueId}">
                            ${dayTitle}
                            ${eventsHtml}
                        </button>
                    </li>
                `;
            }
        }

        return daysHtml;
    }

    /**
     * Calcule les événements pour une date donnée.
     *
     * @param {string} datetime
     * @returns {Object}
     */
    computeEvents(datetime) {
        let eventsHtml = "";
        let nbEvents = "";
        if (this.mode === "events") {
            const startDate = new Date(datetime);
            startDate.setHours(0, 0, 0, 0);

            const endDate = new Date(datetime);
            endDate.setHours(0, 0, 0, 0);

            const events = this.data.filter((event) => new Date(event.start) <= startDate && new Date(event.end) >= endDate);
            const uniqueColors = [...new Set(events.map(event => event.color))];
            eventsHtml = uniqueColors.map(color => `<span class="event" style="--event-color: ${color}"></span>`);


            if (events.length > 0) {
                const nbEventsLabel = pmbDojo.messages.getMessage("calendar", "events_count").replace("!!nbEvents!!", events.length);
                nbEvents = `<span class="events-count">${nbEventsLabel}</span>`;

                // On limite l'affichage des événements à 4
                eventsHtml = `<div class="events">${eventsHtml.slice(0, 4).join("")}</div>`;
            } else {
                const nbEventsLabelNone = pmbDojo.messages.getMessage("calendar", "events_count_none");
                nbEvents = `<span class="events-count">${nbEventsLabelNone}</span>`;
                eventsHtml = `<div class="events"></div>`;
            }
        }

        return { eventsHtml, nbEvents };
    }

    /**
     * Affiche le calendrier.
     *
     * @returns {void}
     */
    renderCalendar() {
        const days = this.generateDays();

        const daysContainer = this.container.querySelector(".calendar-days");
        daysContainer.innerHTML = `<ol data-firstday="${this.getFirstDay()}">${days}</ol>`;

        this.removeEventsContainer();

        this.setupEventsDays();
    }

    /**
     * Change le mois actuel du calendrier en ajoutant ou soustrayant la valeur spécifiée.
     *
     * @param {number} change
     */
    changeMonth(change) {
        this.currentDate.setMonth(this.currentDate.getMonth() + change);

        const direction = change < 0 ? "prev" : "next";
        this.switchAnimation(direction);
    }

    /**
     * Change la direction de l'animation du calendrier.
     *
     * @param {string} direction (default: "next")
     */
    switchAnimation(direction = "next") {
        const calendarDays = this.container.querySelector(".calendar-days");
        const directionClass = direction === "next" ? "month-next" : "month-prev";
        const outClass = `${directionClass}-out`;
        const inClass = `${directionClass}-in`;

        calendarDays.classList.remove("month-prev-out", "month-next-out", "month-prev-in", "month-next-in");
        calendarDays.classList.add(outClass);

        const handleAnimationEnd = () => {
            this.updateSelectedMonth();
            this.updateSelectedYear();
            this.renderCalendar();
            calendarDays.removeEventListener("webkitAnimationEnd", handleAnimationEnd);
            calendarDays.classList.remove(outClass);
            calendarDays.classList.add(inClass);
        };
        calendarDays.addEventListener("webkitAnimationEnd", handleAnimationEnd);
    }

    /**
     * Mise a jour du selecteur des mois
     *
     * @returns {void}
     */
    updateSelectedMonth() {
        const monthSelect = this.container.querySelector(".month-select");
        monthSelect.value = this.currentDate.getMonth();

        this.updateMonthYear();
    }

    /**
     * Mise a jour du selecteur des années
     *
     * @returns {void}
     */
    updateSelectedYear() {
        const yearSelect = this.container.querySelector(".year-select");
        yearSelect.value = this.currentDate.getFullYear();

        this.updateMonthYear();
    }

    /**
     * Mise a jour du datetime de l'élément .month-year avec la date actuelle.
     *
     * @returns {void}
     */
    updateMonthYear() {
        const monthYear = this.container.querySelector(".month-year");
        monthYear.setAttribute("datetime", `${this.currentDate.getFullYear()}-${this.pad(this.currentDate.getMonth() + 1)}`);

        const previousButton = this.container.querySelector(".prev-month");
        const nextButton = this.container.querySelector(".next-month");

        const date = new Date(this.currentDate);
        const previousMonth = date.setMonth(date.getMonth() - 1);
        const nextMonth = date.setMonth(date.getMonth() + 2);

        const longFormatter = new Intl.DateTimeFormat(this.language, {
            month: "long",
            year: "numeric",
        });

        const previousMonthLabel = pmbDojo.messages.getMessage("calendar", "previous_month").replace("!!previous_month!!", longFormatter.format(previousMonth));
        const nextMonthLabel = pmbDojo.messages.getMessage("calendar", "next_month").replace("!!next_month!!", longFormatter.format(nextMonth));

        previousButton.setAttribute("aria-label", previousMonthLabel);
        nextButton.setAttribute("aria-label", nextMonthLabel);

        previousButton.setAttribute("title", previousMonthLabel);
        nextButton.setAttribute("title", nextMonthLabel);
    }

    /**
     * Définit les événements pour le calendrier.
     *
     * @returns {void}
     */
    setupEvents() {
        this.container.querySelector(".prev-month").addEventListener("click", () => this.changeMonth(-1));

        this.container.querySelector(".next-month").addEventListener("click", () => this.changeMonth(1));

        this.container.querySelector(".month-select").addEventListener("change", (e) => {
            const direction = this.currentDate.getMonth() < parseInt(e.target.value) ? "next" : "prev";
            this.currentDate.setMonth(parseInt(e.target.value));

            this.switchAnimation(direction);
        });

        this.container.querySelector(".year-select").addEventListener("change", (e) => {
            const direction = this.currentDate.getFullYear() < parseInt(e.target.value) ? "next" : "prev";
            this.currentDate.setFullYear(parseInt(e.target.value));

            this.switchAnimation(direction);
        });
    }

    /**
     * Définit les événements pour les jours du calendrier.
     *
     * @returns {void}
     */
    setupEventsDays() {
        this.container.querySelectorAll("button.day-button")
            .forEach((dayButton) => dayButton.addEventListener("click", this.clickDay.bind(this, dayButton)));
    }

    /**
     * Renvoie les abbreviations et les noms des jours de la semaine, dans l'ordre et la langue définis par l'attribut "language".
     *
     * @returns {Array<{abbr: string, label: string}>}
     */
    getWeekdays() {
        const shortFormatter = new Intl.DateTimeFormat(this.language, {
            weekday: "short",
        });
        const longFormatter = new Intl.DateTimeFormat(this.language, {
            weekday: "long",
        });
        const weekdays = [];
        const startDate = new Date(2023, 0, 2); // Commence à Lundi

        for (let i = 0; i < 7; i++) {
            const currentDate = new Date(startDate);
            currentDate.setDate(startDate.getDate() + i);

            const shortDay = shortFormatter.format(currentDate);
            const longDay = longFormatter.format(currentDate);

            weekdays.push({
                abbr: shortDay.charAt(0).toUpperCase() + shortDay.slice(1),
                label: longDay.charAt(0).toUpperCase() + longDay.slice(1),
            });
        }

        return weekdays;
    }

    /**
     * Retourne un tableau des 12 mois de l'année, dans l'ordre et la langue définis par "language".
     *
     * @returns {string[]}
     */
    getMonths() {
        const monthFormatter = new Intl.DateTimeFormat(this.language, {
            month: "long",
        });
        const months = [];
        for (let i = 0; i < 12; i++) {
            const date = new Date(2023, i, 1);
            months.push(monthFormatter.format(date).charAt(0).toUpperCase() + monthFormatter.format(date).slice(1));
        }
        return months;
    }

    /**
     * Retourne un tableau des 20 années centrées sur l'année actuelle.
     *
     * @returns {number[]}
     */
    getYears() {
        const years = [];
        const currentYear = this.currentDate.getFullYear();
        if (this.mode === "events" && this.data.length > 0) {
            const minYear = this.data.reduce((min, event) => Math.min(min, new Date(event.start).getFullYear(), new Date().getFullYear()), Infinity);
            const maxYear = this.data.reduce((max, event) => Math.max(max, new Date(event.end).getFullYear(), new Date().getFullYear()), -Infinity);
            for (let year = minYear; year <= maxYear; year++) {
                years.push(year);
            }
            return years;
        }

        for (let year = currentYear - 10; year <= currentYear + 10; year++) {
            years.push(year);
        }
        return years;
    }

    /**
     * Calcule le numéro de la semaine pour une date donnée.
     *
     * @param {Date|string} date
     * @returns {number}
     */
    getWeekNumber(date) {
        const d = new Date(date);

        d.setHours(0, 0, 0, 0);
        d.setDate(d.getDate() + 4 - (d.getDay() || 7));

        const yearStart = new Date(d.getFullYear(), 0, 1);
        return Math.ceil(((d - yearStart) / 86400000 + 1) / 7);
    }

    /**
     * Retourne le numéro du premier jour du mois courant.
     * Les jours de la semaine sont numérotés de 1 (lundi) à 7 (dimanche).
     *
     * @returns {number}
     */
    getFirstDay() {
        const firstDay = new Date(this.currentDate.getFullYear(), this.currentDate.getMonth(), 1);
        const day = firstDay.getDay();
        if (0 === day) {
            return 7;
        }
        return day;
    }

    /**
     * Obtient la langue actuellement définie pour la page.
     * Si aucune langue n'est définie, retourne 'fr-FR' par défaut.
     *
     * @returns {string}
     */
    get language() {
        if (!this._language) {
            this._language = document.documentElement.getAttribute("lang") || navigator.language || "fr-FR";
        }

        return this._language;
    }

    /**
     * Définit la langue pour le calendrier.
     *
     * @param {string} language
     * @param {boolean} buildRender
     * @returns {void}
     */
    setLanguage(language, buildRender = true) {
        try {
            new Intl.DateTimeFormat(language);
            this._language = language;
        } catch (e) {
            console.error(`La langue "${language}" n'est pas supportée.`);
            this.setLanguage(navigator.language);
        }

        if (buildRender) {
            this.buildRender();
        }
    }

    /**
     * Définit le conteneur du calendrier en fonction de son identifiant.
     *
     * @param {string} containerId
     * @returns {void}
     */
    setContainer(containerId) {
        this.container = document.getElementById(containerId);

        if (!this.container) {
            console.error("Le conteneur n'existe pas.");
        }
    }

    /**
     * Définit la date actuelle du calendrier.
     *
     * @param {string|Date} date
     * @param {boolean} buildRender
     * @returns {void}
     */
    setDate(date, buildRender = true) {
        this.currentDate = new Date(date);

        if (buildRender) {
            this.buildRender();
        }
    }

    /**
     * Définit le mode du calendrier.
     *
     * @param {number} mode
     * @param {boolean} buildRender
     * @returns {void}
     */
    setMode(mode, buildRender = true) {
        switch (mode) {
            case "events":
            case "datePicker":
                this.mode = mode;
                break;
            default:
                console.error("Le mode n'est pas supporté.");
                this.mode = "datePicker";
        }

        if (this.mode === "datePicker") {
            this.data = [];
        }
        if (this.mode === "events") {
            this.onSelectedDate = () => {};
        }

        if (buildRender) {
            this.buildRender();
        }
    }

    /**
     * Définit les données supplémentaires du calendrier pour le mode spécifique.
     *
     * @param {Array} data
     * @param {boolean} buildRender
     * @returns {void}
     */
    setData(data, buildRender = true) {
        if (!Array.isArray(data)) {
            console.error("Les données doivent être un tableau.");
            return;
        }

        this.data = data;
        const found = this.data.some((e) => {
            const startDate = new Date(e.start);
            const endDate = new Date(e.end);

            return (
                startDate.getFullYear() <= this.currentDate.getFullYear() &&
                endDate.getFullYear() >= this.currentDate.getFullYear() &&
                startDate.getMonth() <= this.currentDate.getMonth() &&
                endDate.getMonth() >= this.currentDate.getMonth()
            );
        });

        if (this.data.length === 0 || !found) {
            this.setDate(new Date(), false);
        }

        if (buildRender) {
            this.buildRender();
        }
    }

    /**
     * Retourne une chaîne de caractères en ajoutant un zéro à gauche si la valeur est inférieure à 10.
     *
     * @param {number} val
     * @returns {string}
     */
    pad(val) {
        return val.toString().padStart(2, "0");
    }

    /**
     * Gère le clic sur un jour du calendrier.
     *
     * @param {HTMLElement} button
     * @param {Event} e
     * @returns {void}
     */
    clickDay(button, e) {
        switch (this.mode) {
            case "events":
                this.clickDayEvents(button, e);
                break;

            case "datePicker":
                this.clickDayDatePicker(button, e);
                break;

            default:
                console.error("Le mode n'est pas supporté.");
                break;
        }
    }

    /**
     * Gère le clic sur un jour du calendrier en mode "datePicker".
     *
     * @param {HTMLElement} button
     * @param {Event} e
     * @returns {void}
     */
    clickDayDatePicker(button, e) {
        const date = button.querySelector("time").getAttribute("datetime");
        this.onSelectedDate(date, e);
    }

    /**
     * Gère le clic sur un jour du calendrier en mode "events".
     *
     * @param {HTMLElement} button
     * @param {Event} e
     * @returns {void}
     */
    clickDayEvents(button, e) {
        const calendarContainer = this.container.querySelector(".calendar");
        if (calendarContainer.querySelector("div[aria-labelledby='" + button.getAttribute("id") + "'].events-container")) {
            this.removeEventsContainer();
            return;
        }

        this.removeEventsContainer();
        const events = this.data.filter((event) => {
            const time = new Date(button.querySelector("time").getAttribute("datetime"));
            time.setHours(0, 0, 0, 0);

            return new Date(event.start) <= time && new Date(event.end) >= time;
        });

        if (events.length === 0) {
            return;
        }

        const position = this.computePosition(button, calendarContainer);
        let eventsContainer = document.createElement("div");

        eventsContainer.style.top = position.top + "px";
        eventsContainer.classList.add("events-container", "fade-events");
        eventsContainer.id = "events_container_" + Date.now();
        eventsContainer.setAttribute("role", "dialog");
        eventsContainer.setAttribute("tabindex", "0");
        eventsContainer.setAttribute("aria-labelledby", button.getAttribute("id"));
        button.setAttribute("aria-describedby", eventsContainer.id);

        const liList = events.map((event) => {
            const template = `<li class="event-item"><span class="event" style="--event-color: ${event.color}"></span><a href="${event.link}">!!title!!</a></li>`;

            const startDate = new Date(event.start);
            const startDateStr = this.printableDate(startDate);

            const endDate = new Date(event.end);
            const endDateStr = this.printableDate(endDate);

            const title = `${event.title} (${startDateStr} - ${endDateStr})`;
            return template.replace("!!title!!", title);
        });

        const eventsHtml = `<ul>${liList.join("")}</ul>`;
        eventsContainer.innerHTML = eventsHtml;

        const closeButton = document.createElement("button");
        closeButton.classList.add("close-button");
        closeButton.innerHTML = "&times;";
        closeButton.setAttribute("aria-label", pmbDojo.messages.getMessage("calendar", "close_events_dialog"));
        closeButton.setAttribute("title", pmbDojo.messages.getMessage("calendar", "close_events_dialog"));
        closeButton.setAttribute("tabindex", "0");
        closeButton.addEventListener("click", () => this.removeEventsContainer());

        eventsContainer.appendChild(closeButton);

        const arrow = document.createElement("div");
        arrow.classList.add("arrow", "fade-events");
        calendarContainer.appendChild(arrow);

        arrow.style.setProperty("top", (position.top - arrow.offsetHeight) + "px");
        arrow.style.setProperty("left", (position.left - (arrow.offsetWidth / 2)) + "px");

        eventsContainer.addEventListener("keydown", (e) => {
            if (e.key === "Escape") {
                this.removeEventsContainer();
                button.focus();
            }
        });
        eventsContainer.addEventListener("blur", (e) => {
            if (!eventsContainer.contains(e.relatedTarget)) {
                this.removeEventsContainer();
                button.focus();
            }
        });

        const calendarDaysContainer = this.container.querySelector(".calendar-days");
        calendarDaysContainer.addEventListener("click", (e) => e.stopPropagation());
        window.addEventListener("click", () => this.removeEventsContainer());

        calendarContainer.appendChild(eventsContainer);
        eventsContainer.focus();
        focus_trap(eventsContainer);
    }

    computePosition(node, container, isParent = false) {
        let top = node.offsetTop;
        let left = node.offsetLeft;
        if (!isParent) {
            top += node.offsetHeight;
            left += (node.offsetWidth / 2);
        }

        if (node.offsetParent !== container) {
            const position = this.computePosition(node.offsetParent, container, true);
            top += position.top;
            left += position.left;
        }

        return {top, left};
    }

    /**
     * Supprime le conteneur d'événements.
     *
     * @returns {void}
     */
    removeEventsContainer() {
        const buttonList = this.container.querySelectorAll(".day-button");
        buttonList.forEach((button) => button.removeAttribute("aria-describedby"));

        const eventsContainerList = this.container.querySelectorAll(".events-container");
        eventsContainerList.forEach((eventsContainer) => eventsContainer.remove());

        const arrowList = this.container.querySelectorAll(".arrow");
        arrowList.forEach((arrow) => arrow.remove());

        document.onmousedown = null;
    }

    /**
     * Retourne une date formatée en fonction du format de l'application.
     *
     * @param {Date} date
     * @returns {string}
     */
    printableDate(date) {
        let options = {
            year: "numeric",
            month: "numeric",
            day: "numeric",
        };

        if (date.getHours() !== 0 || date.getMinutes() !== 0) {
            options.hour = "numeric";
            options.minute = "numeric";
        }

        const longFormatter = new Intl.DateTimeFormat(this.language, options);
        return longFormatter.format(date);
    }

    /**
     * Gère la sélection d'une date.
     *
     * @param {string} date
     * @param {Event} e
     * @returns {void}
     */
    onSelectedDate = (date, e) => {};
}
