<?php

// +-------------------------------------------------+
//  2002-2004 PMB Services / www.sigb.net pmb@sigb.net et contributeurs (voir www.sigb.net)
// +-------------------------------------------------+
// $Id: RootThumbnailSource.php,v 1.18.4.1 2025/09/05 09:47:18 qvarin Exp $

namespace Pmb\Thumbnail\Models\Sources;

use Pmb\Common\Helper\GlobalContext;
use Pmb\Common\Models\Model;
use Pmb\Thumbnail\Orm\SourcesORM;
use Pmb\Common\Helper\ParserMessage;
use Pmb\Thumbnail\Models\Sources\Entities\Record\Noimage\NoImageThumbnailSource;

class RootThumbnailSource extends Model
{
    use ParserMessage;

    public const CACHE_HEADER = 'X-Thumbnail-Cache';
    public const SOURCE_HEADER = 'X-Thumbnail-Source';

    public const NO_IMAGE_SOURCES = [
        NoImageThumbnailSource::class,
    ];

    /**
     * taille minimale de l'image
     * @var int
     */
    public const IMG_MIN_SIZE = 100;

    /**
     * taille de l'image
     * @var int
     */
    public const IMG_MAX_SIZE = 1024 * 1024;

    /**
     * temps d'attente en seconde avant expiration du curl
     * @var int
     */
    public const CURL_TIMEOUT = 5;

    /**
     * orm utilise pour la manipulation de la base de donnes
     * @var string
     */
    protected $ormName = "Pmb\\Thumbnail\\Orm\\SourcesORM";

    /**
     * paramterage de la source
     * @var array
     */
    protected $settings = [];

    /**
     * source activee ou non
     * @var int
     */
    protected $active = 1;

    /**
     * tableau des headers
     * @var array
     */
    protected $imageHeaders = [];

    /**
     * source autorisant le cache
     * @var bool
     */
    protected $allowedCache = true;

    private function __construct(int $id = 0)
    {
        parent::__construct($id);
    }

    /**
     *
     * @return \Pmb\Thumbnail\Models\Sources\RootThumbnailSource
     */
    public static function getInstance(): RootThumbnailSource
    {
        $resuts = SourcesORM::find("class", static::class);
        $id = ! empty($resuts) ? $resuts[0]->id : 0;
        return new static($id);
    }

    /**
     * recuperation des parametres
     * @return array
     */
    public function getParameters(): array
    {
        return $this->settings;
    }

    /**
     * definition des parametres
     * @param array $settings
     */
    public function setParameters(array $settings): void
    {
        $this->settings = $settings;
    }

    /**
     * retourne le contenu de l'image
     * @param int $object_id
     * @return string
     */
    public function getImage(int $object_id): string
    {
        return "";
    }

    /**
     * sauvegarde de la source
     * @return bool
     */
    public function save(): bool
    {
        $orm = new $this->ormName($this->id);
        $orm->class = static::class;
        $orm->settings = \encoding_normalize::json_encode($this->settings);
        $orm->active = true;
        $orm->save();

        $this->id = $orm->id;
        return ! empty($this->id) && $this->id != 0;
    }

    /**
     * telechargement de l'image distante avec curl
     * @param string $image_url
     * @param bool $returnNoImage
     * @return string
     */
    protected function loadImageWithCurl(string $image_url, bool $returnNoImage = true): string
    {
        $curl = new \Curl();
        $curl->limit = RootThumbnailSource::IMG_MAX_SIZE;
        $curl->timeout = $this->settings["curl_timeout"] ?? RootThumbnailSource::CURL_TIMEOUT;
        $curl->options['CURLOPT_SSL_VERIFYPEER'] = 0;
        $curl->options['CURLOPT_ENCODING'] = '';

        $content = $curl->get($image_url);
        if (empty($content)) {
            return '';
        }

        if ($content->headers['Status-Code'] != 200 || empty($content->body)) {
            return '';
        }

        $image = $content->body;
        if (empty($content->headers['Content-Length']) && strlen($image)) {
            $content->headers['Content-Length'] = strlen($image);
        } else {
            $content->headers['Content-Length'] = intval($content->headers['Content-Length']);
        }

        if (!$this->hasRequiredSize($content->headers['Content-Length'])) {
            return '';
        }

        if (
            $returnNoImage === false &&
            $this->isThumbnailUrl($image_url) &&
            $this->isNoImageSource($content->headers[RootThumbnailSource::SOURCE_HEADER] ?? '')
        ) {
            return '';
        }

        $this->imageHeaders = $content->headers;
        return $image;
    }

    /**
     * source active
     * @return bool
     */
    public function isActive(): bool
    {
        if (!empty($this->active)) {
            return true;
        }
        return false;
    }

    /**
     * recuperation du watermark
     * @return string
     */
    public function getWatermark(): string
    {
        return $this->settings["watermark"] ?? "";
    }

    /**
     * recuperation des headers
     * @return array
     */
    public function getImageHeaders(): array
    {
        return $this->imageHeaders;
    }

    /**
     * source ayant autorise le cache
     * @return bool
     */
    public function hasAllowedCache(): bool
    {
        return $this->allowedCache;
    }

    /**
     * Verifie si la taille est dans le bon intervalle
     *
     * @param int $size
     * @return boolean
     */
    protected function hasRequiredSize(int $size): bool
    {
    	return $size <= RootThumbnailSource::IMG_MAX_SIZE
    	&& $size >= RootThumbnailSource::IMG_MIN_SIZE;
    }

    /**
     * On est sur une url de pmb
     *
     * @param string $url
     * @return boolean
     */
    protected function isInternalUrl(string $url): bool
    {
        return strpos($url, GlobalContext::get("opac_url_base")) === 0
            || strpos($url, GlobalContext::get("pmb_url_base")) === 0
            || strpos($url, GlobalContext::get("pmb_url_internal")) === 0;
    }

    /**
     * On est sur une url de thumbnail
     *
     * @param string $url
     * @return boolean
     */
    protected function isThumbnailUrl(string $url): bool
    {
        return $this->isInternalUrl($url)
            && strpos(parse_url($url, PHP_URL_PATH), '/thumbnail.php') !== false;
    }

    /**
     * On est sur une source no image
     *
     * @param string $header
     * @return boolean
     */
    protected function isNoImageSource(string $header): bool
    {
        if (empty($header)) {
            return false;
        }
        return in_array($header, RootThumbnailSource::NO_IMAGE_SOURCES);
    }
}
